/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Hashtable;
import java.util.Vector;

/**
 * The model for the Output View, containing actions, parameter values,
 * and result sets. 
 */
public class OutputViewModel
{
   /** A Hashtable of OutputObjects, keyed by OutputItem. */
   protected Hashtable outputObjects;
   /** A Vector of actions. */
   protected Vector actionList;
   /** A model for the current result set table. */
   protected ResultTableModel currentResultModel;

   /** Constructs a model. */
   public OutputViewModel()
   {
      outputObjects = new Hashtable();
      actionList = new Vector();
      currentResultModel = new ResultTableModel();
   }

   /**
    *  Adds a new outputItem to the actionList in the model.
    *  @param item An OutputItem to add.
    */
   protected void addOutputItem(OutputItem item)
   {
      if (!actionList.contains(item))
      {
         actionList.add(0,item);     // add to the beginning
         outputObjects.put(item, new OutputObject());
      }
   }

   /**
    * Determines whether an item exits already in the model.
    * @param item An OutputItem to check.
    * @return True if the item exists.
    */
   protected boolean outputItemExists(OutputItem item)
   {
      return  actionList.contains(item);
   }

   /**
    * Determines whether the action list is empty.
    * @return True if the list is empty.
    */
   protected boolean isActionListEmpty()
   {
      return actionList.isEmpty();
   }

   /**
    * Gets the current result set table model.
    * @return A model for the current result set table.
    */
   protected ResultTableModel getResultTableModel()
   {
      return currentResultModel;
   }

   /**
    * Gets the  result set table model for a given item.
    * @param item An OutputItem.
    * @return A model for the result set table.
    */
   protected ResultTableModel getResultTableModel(OutputItem item)
   {
      if (!outputObjects.contains(item))
      {
         OutputObject ob = (OutputObject)outputObjects.get(item);
         return ob.getResultTableModel();
      }
      else
         return null;
   }

   /**
    * Gets the number of rows in the result set for a given item.
    * @param item An OutputItem.
    * @return The row count.
    */
   protected int getResultSetRowCount(OutputItem item)
   {
      ResultTableModel rsModel = getResultTableModel(item);
      if (rsModel == null)
         return -1;
      else
         return rsModel.getResultSetRowCount();
   }

   /**
    * Adds parameters for a given item.
    * @param item An OutputItem.
    * @param parms Parameters to be added.
    */
   protected void addParameters(OutputItem item, Vector parms)
   {
      addOutputItem(item);
      if (parms != null)
         item.setHasParameters(true);
      OutputObject ob = (OutputObject)outputObjects.get(item);
      ob.setParameters(parms);
   }

   /**
    * Gets parameters for a given item.
    * @param item An OutputItem.
    * @return Parameters for the item.
    */
   protected Vector getParameters(OutputItem item)
   {
      if (!outputObjects.contains(item))
      {
         OutputObject ob = (OutputObject)outputObjects.get(item);
         return ob.getParameters();
      }
      else
         return null;
   }

   /**
    * Adds an item for a procedure call, with results.
    * @param item An OutputItem to be added.
    * @param proc The cache for the call results.
    */
   protected void addProcedureCall(OutputItem item, ProcedureCallCache proc) throws SQLException
   {
      addOutputItem(item);
      OutputObject ob = (OutputObject)outputObjects.get(item);
      ob.setProcedureCall(proc);
      if (proc != null)
         item.setHasResults(true);
   }

   /**
    * Adds an item for a procedure call with results and parameters.
    * @param item An OutputItem to be added.
    * @param proc The cache for the call results.
    * @param parms The parameters.
    */
   protected void addProcedureCallAndParameters(OutputItem item, ProcedureCallCache proc, Vector parms) throws SQLException
   {
      addOutputItem(item);
      OutputObject ob = (OutputObject)outputObjects.get(item);
      ob.setProcedureCall(proc);
      item.setHasResults(true);
      if (parms != null)
      {
         item.setHasParameters(true);
         ob.setParameters(parms);
      }
   }

   /**
    * Adds results to an item.
    * @param item An OutputItem.
    * @param rs The result set to be added.
    */
   protected void addResult(OutputItem item, ResultSet rs) throws SQLException
   {
      addResultAndParameters( item, rs, null );
   }

   /**
    * Adds results and parameters to an item.
    * @param item An OutputItem.
    * @param rs The result set to be added.
    * @param parms The parameters to be added.
    */
   protected void addResultAndParameters(OutputItem item, ResultSet rs, Vector parms) throws SQLException
   {
      addOutputItem(item);
      OutputObject ob = (OutputObject) outputObjects.get(item);
      if (parms != null) {
         item.setHasParameters(true);
         ob.setParameters(parms);
      }

      // If the resultset is not viable, addTableModel will throw an SQLException.
      // For example, if a udf wrapped a soap UDF that made a call to a
      // remote site and the connection was down, then the first we
      // see a problem is when we try to access the result set.  We'd get an exception like:
      // [IBM][CLI Driver][DB2/NT] SQL0443N  Routine "DB2XML.SOAPHTTPV" (specific name "SOAPHTTPVIVO")
      // has returned an error SQLSTATE with diagnostic text "Error resolve host name".  SQLSTATE=38309
      ob.addTableModel(rs);
      if (rs != null )
         item.setHasResults(true);
   }

   /**
    * Gets the action list.
    * @return A vector of actions.
    */
   protected Vector getActionList()
   {
      return actionList;
   }

   /**
    * Determines whether the first result set for an item is selected.
    * @param item An OutputItem.
    * @return True if the first result set for an item is selected.
    */
   protected boolean isFirstResultSet(OutputItem item)
   {
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
         return ob.isFirstResultSet();
      else
         return true;
   }

   /**
    * Determines whether the last result set for an item is selected.
    * @param item An OutputItem.
    * @return True if the last result set for an item is selected.
    */
   protected boolean isLastResultSet(OutputItem item)
   {
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
         return ob.isLastResultSet();
      else
         return true;
   }

   /**
    * Gets the next result set model for an item.
    * @param item An OutputItem.
    * @return  The next result set model for an item, or null if this is the last.
    */
   protected ResultTableModel getNextResultSetModel(OutputItem item) throws SQLException
   {
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
         return ob.getNextResultSetModel();
      else
         return null;
   }

   /**
    * Gets the previous result set model for an item.
    * @param item An OutputItem.
    * @return  The previous result set model for an item, or null if this is the first.
    */
   protected ResultTableModel getPreviousResultSetModel(OutputItem item) throws SQLException
   {
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
         return ob.getPreviousResultSetModel();
      else
         return null;
   }

   /**
    * Gets an item by index.
    * @param idx The index.
    * @return The item.
    */
   protected OutputItem getItem(int idx)
   {
      if (idx >= 0)
         return (OutputItem)actionList.get(idx);
      else
         return null;
   }

   /**
    * Removes all records.
    */
   protected void removeAll()
   {
      outputObjects.clear();
      actionList.clear();
   }

   /**
    * Removes records for a given item.
    * @param item An OutputItem.
    */
   protected void removeOutputItem(OutputItem item)
   {
      if (actionList.contains(item))
      {
         outputObjects.remove(item);
         actionList.remove(item);
      }
   }

   /**
    * Removes records for a given index.
    * @param index The index of an OutputItem.
    */
   protected void removeOutputItem(int index)
   {
      outputObjects.remove( actionList.get(index));
      actionList.remove(index);
   }

   /**
    * Resets records for a given item and returns the item.
    * @param item An OutputItem.
    * @return The item, after being reset.
    */
   protected OutputItem resetItem(OutputItem item)
   {
      item.setMessage("");
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
      {
         ob.init();
         item.setHasResults(false);
         item.setHasParameters(false);
      }
      return item;
   }

   /**
    * Determines whether an item has multiple result sets.
    * @param item An OutputItem.
    * @return True if an item has multiple result sets.
    */
   protected boolean multipleResults(OutputItem item)
   {
      OutputObject ob = (OutputObject)outputObjects.get(item);
      if (ob != null)
         return ob.multipleResults();
      else
         return false;
   }

   /**
    * Finds an item by unique name and action code.
    * @param uniqueName The unique name for the item.
    * @param action The action code.
    * @return The item, or null if it cannot be found.
    */
   protected OutputItem findOutputItem(String uniqueName, int action)
   {
      OutputItem item = null;
      if (uniqueName != null)
      {
         for (int i=0; i < actionList.size(); i++)
         {
            item = (OutputItem)actionList.get(i);
            if ( item.getUniqueName().equals(uniqueName) &&
                     item.getActionCode() == action )
            {
               return item;
            }
         }
      }
      return null;
   }

   /**
    * Associates all information about an item.
    */
   protected class OutputObject
   {
      /** The item's procedure call information. */
      protected ProcedureCallCache procedureCall;
      /** A Vector of result set table models. */
      protected Vector resultTableModels;
      /** True if the item has multiple result sets. */
      protected boolean multipleResultSets;
      /** True if the item represents a procedure call. */
      protected boolean isProcCall;
      /** A Vector of parameter information. */
      protected Vector parameters;
      /* The current result set.
      protected ResultSet rs; */
      /** The current table model. */
      protected int currentTabModel;
      /** Construct an output object. */
      protected OutputObject()
      {
         init();
      }
      /** Initializes the object. */
      protected void init()
      {
         procedureCall = null;
         parameters = null;
         //rs = null;
         multipleResultSets = false;
         isProcCall = false;
         resultTableModels = new Vector();
         currentTabModel = 0;
      }
      /**
       * Gets the current result set table model.
       * @return The current result set table model.
       */
      protected ResultTableModel getResultTableModel()
      {
         if (resultTableModels.isEmpty())
            return null;
         else
            return (ResultTableModel)resultTableModels.get(currentTabModel);
      }
      /**
       * Gets the procedure call information.
       * @return The procedure call information.
       */
      protected ProcedureCallCache getProcedureCall()
      {
         return procedureCall;
      }
      /**
       * Gets the parameter Vector.
       * @return The parameter Vector.
       */
      protected Vector getParameters()
      {
         return parameters;
      }
      /**
       * @return True if this item has multiple result sets.
       */
      protected boolean multipleResults()
      {
         return multipleResultSets;
      }
      /**
       * @return True if the first result sets is selected.
       */
      protected boolean isFirstResultSet()
      {
         if (procedureCall != null)
         {
            //  return (procedureCall.getCurrentResult() == 1);
            return (currentTabModel == 0);
         }
         else
            return true;
      }
      /**
       * @return True if the last result sets is selected.
       */
      protected boolean isLastResultSet()
      {
         if (procedureCall != null)
         {
            // if there is a next in the cache, it is not the last
            //   otherwise check in procedureCall
            if (currentTabModel+1 < resultTableModels.size())
               return false;
            else
               return (procedureCall.isLastResult());
         }
         else
            return true;
      }
      /**
       * Gets the next result set table model.
       * @return The next result set table model.
       */
      protected ResultTableModel getNextResultSetModel() throws SQLException
      {
         if (procedureCall != null)
         {
            boolean getFromDB = false;
            // first check if result set table model is cached
            if ( (currentTabModel+1 < resultTableModels.size() ) &&
                     (resultTableModels.get(currentTabModel+1) != null))
            {
               currentTabModel = currentTabModel + 1;
            }
            else //not cached, get it from procedureCall
            {
               if (!procedureCall.isLastResult())
               {
                  getNextResultSet();
                  resultTableModels.add(new ResultTableModel(procedureCall));
                  currentTabModel = currentTabModel + 1;
               }
            }
         }
         return getResultTableModel();
      }
      /**
       * Gets the previous result set table model.
       * @return The next result set table model.
       */
      protected ResultTableModel getPreviousResultSetModel() throws SQLException
      {
         if (procedureCall != null)
         {
            // first check if result set table model is cached
            if ( ( currentTabModel-1 >= 0 ) &&
                     (resultTableModels.get(currentTabModel-1) != null) )
               currentTabModel = currentTabModel - 1;
            else //not cached, get it from procedureCall(not likely, as we traverse each one)
            {
               if (!(procedureCall.getCurrentResult() == 1) )
               {
                  getPrevResultSet();
                  resultTableModels.add(new ResultTableModel(procedureCall));
                  currentTabModel = currentTabModel - 1;
               }
            }
         }
         return getResultTableModel();
      }
      /**
       * Sets the procedure call information.
       * @param pc The ProcedureCallCache for this item.
       */
      protected void setProcedureCall(ProcedureCallCache pc) throws SQLException
      {
         procedureCall = pc;
         isProcCall = true;
         multipleResultSets = !procedureCall.isLastResult();
         resultTableModels.add(new ResultTableModel(procedureCall));
      }
      /**
       * Adds a result set table model.
       * @param rs A ResultSet for this item.
       */
      protected void addTableModel(ResultSet rs) throws SQLException
      {
         resultTableModels.add(new ResultTableModel(rs));
      }
      /**
       * Sets the parameter vector.
       * @param parameters The parameter information for this item.
       */
      protected void setParameters(Vector parameters)
      {
         this.parameters = parameters;
      }
      /**
       * Moves to the previous result set.
       * @return The ProcedureCallCache set to the previous result set.
       */
      protected ProcedureCallCache getPrevResultSet() throws SQLException
      {
         procedureCall.previousResult();
         return procedureCall;
      }
      /**
       * Moves to the next result set.
       * @return The ProcedureCallCache set to the next result set.
       */
      protected ProcedureCallCache getNextResultSet() throws SQLException
      {
         procedureCall.nextResult();
         return procedureCall;
      }
   }

}
