/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.navigator.internal.provisional.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.wst.common.navigator.internal.views.NavigatorPlugin;
import org.eclipse.wst.common.navigator.internal.views.actions.CommonActionProviderDescriptor;
import org.eclipse.wst.common.navigator.internal.views.extensions.INavigatorContentServiceListener;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptorInstance;
import org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptorRegistry;
import org.eclipse.wst.common.navigator.internal.views.extensions.RegistryReader;
import org.eclipse.wst.common.navigator.internal.views.extensions.SkeletonActionProvider;

/**
 * <p>
 * The following class is experimental until fully documented.
 * </p>
 */
public class NavigatorActionService implements INavigatorContentServiceListener {


	private static final NavigatorContentDescriptorRegistry CONTENT_DESCRIPTOR_REGISTRY = NavigatorContentDescriptorRegistry.getInstance();

	private static final CommonActionProviderDescriptor[] NO_DESCRIPTORS = new CommonActionProviderDescriptor[0];

	private static final IContributionItem NEW_GROUP = new Separator(ICommonMenuConstants.GROUP_NEW);
	private static final IContributionItem GOTO_GROUP = new GroupMarker(ICommonMenuConstants.GROUP_GOTO);
	private static final IContributionItem OPEN_GROUP = new Separator(ICommonMenuConstants.GROUP_OPEN);
	private static final IContributionItem SHOW_GROUP = new GroupMarker(ICommonMenuConstants.GROUP_SHOW);
	private static final IContributionItem EDIT_GROUP = new Separator(ICommonMenuConstants.COMMON_MENU_EDIT_ACTIONS);
	private static final IContributionItem REORGANIZE_GROUP = new GroupMarker(ICommonMenuConstants.GROUP_REORGANIZE);
	private static final IContributionItem PORT_GROUP = new GroupMarker(ICommonMenuConstants.GROUP_PORT);
	private static final IContributionItem GENERATE_GROUP = new Separator(ICommonMenuConstants.GROUP_GENERATE);
	private static final IContributionItem SEARCH_GROUP = new Separator(ICommonMenuConstants.GROUP_SEARCH);
	private static final IContributionItem BUILD_GROUP = new Separator(ICommonMenuConstants.GROUP_BUILD);
	private static final IContributionItem ADDITIONS_GROUP = new Separator(ICommonMenuConstants.GROUP_ADDITIONS);
	private static final IContributionItem VIEWER_GROUP = new Separator(ICommonMenuConstants.GROUP_VIEWER_SETUP);
	private static final IContributionItem PROPERTIES_GROUP = new Separator(ICommonMenuConstants.GROUP_PROPERTIES);

	private final IViewPart viewPart;
	private final StructuredViewer structuredViewer;
	private final NavigatorContentService contentService;
	private final Set actionProviderDescriptors = new HashSet();
	private final Map actionProviderInstances = new HashMap();
	private boolean isApplyingActionsFromContentExtensions;

	private IMemento memento;

	private boolean isDisposed = false; 
	private boolean isInitialized = false;

	/**
	 *  
	 */
	public NavigatorActionService(IViewPart aViewPart, StructuredViewer aStructuredViewer, NavigatorContentService aContentService) {
		this(aViewPart, aStructuredViewer, aContentService, true);
	}

	/**
	 *  
	 */
	public NavigatorActionService(IViewPart aViewPart, StructuredViewer aStructuredViewer, NavigatorContentService aContentService, boolean toApplyActionsFromContentExtensions) {
		super();
		viewPart = aViewPart;
		contentService = aContentService;
		structuredViewer = aStructuredViewer;
		isApplyingActionsFromContentExtensions = toApplyActionsFromContentExtensions;

		init();
	}

	private void init() {
		if(isInitialized)
			return;
		try {
		new CommonActionRegistry().readRegistry();

		Collection contentDescriptors = contentService.getDescriptorInstances();
		if (contentDescriptors.size() > 0) {
			ICommonActionProvider provider = null;
			NavigatorContentDescriptorInstance descriptorInstance = null;
			for (Iterator iter = contentDescriptors.iterator(); iter.hasNext();) {
				descriptorInstance = (NavigatorContentDescriptorInstance) iter.next();
				provider = descriptorInstance.getActionProvider();
				initialize(provider);
			}
		}

		contentService.addListener(this);
		} catch(RuntimeException re) {
			NavigatorPlugin.log("Could not initialize NavigatorActionService for " + contentService.getViewerId() + ": " + re.getMessage());
			re.printStackTrace();
		}
		isInitialized = true;
	}

	public void fillContextMenu(IMenuManager aMenu, IStructuredSelection aStructuredSelection) {
		complainIfDisposed();

		aMenu.add(NEW_GROUP);
		aMenu.add(GOTO_GROUP);
		aMenu.add(OPEN_GROUP);
		aMenu.add(SHOW_GROUP);
		aMenu.add(EDIT_GROUP);
		aMenu.add(REORGANIZE_GROUP);
		aMenu.add(PORT_GROUP);
		aMenu.add(GENERATE_GROUP);
		aMenu.add(SEARCH_GROUP);
		aMenu.add(BUILD_GROUP);
		aMenu.add(ADDITIONS_GROUP);
		aMenu.add(VIEWER_GROUP);
		aMenu.add(PROPERTIES_GROUP);

		if (aStructuredSelection == null || aStructuredSelection.isEmpty())
			aStructuredSelection = new StructuredSelection(structuredViewer.getInput());
		ActionContext context = new ActionContext(aStructuredSelection);
		addContentDescriptorMenu(aMenu, aStructuredSelection, context);
		addCommonActionProviderMenu(aMenu, aStructuredSelection, context);

		aMenu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * @param aMenu
	 * @param aStructuredSelection
	 * @param aContext
	 */
	private void addCommonActionProviderMenu(IMenuManager aMenu, IStructuredSelection aStructuredSelection, ActionContext aContext) {
		CommonActionProviderDescriptor[] providerDescriptors = findRelevantActionDescriptors(aStructuredSelection);
		if (providerDescriptors.length > 0) {
			ICommonActionProvider provider = null;
			for (int i = 0; i < providerDescriptors.length; i++) {
				try {
					provider = getActionProviderInstance(providerDescriptors[i]);
					provider.setActionContext(aContext);
					provider.fillContextMenu(aMenu);
				} catch (RuntimeException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * @param aMenu
	 * @param aStructuredSelection
	 * @param aContext
	 */
	private void addContentDescriptorMenu(IMenuManager aMenu, IStructuredSelection aStructuredSelection, ActionContext aContext) {
		if (isApplyingActionsFromContentExtensions) {
			NavigatorContentDescriptorInstance[] contentDescriptorInstances = contentService.findRelevantContentDescriptorInstances(aStructuredSelection);
			NavigatorContentDescriptorInstance contentDescriptorInstance = null;
			ICommonActionProvider provider = null;
			for (int x = 0; x < contentDescriptorInstances.length; ++x) {
				try {
					contentDescriptorInstance= contentDescriptorInstances[x];
					provider = contentDescriptorInstance.getActionProvider();
					if (provider != SkeletonActionProvider.INSTANCE) {
						provider.setActionContext(aContext);
						provider.fillContextMenu(aMenu);
					}
				} catch (RuntimeException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}

	public void fillActionBars(IActionBars theActionBars, IStructuredSelection aStructuredSelection) {
		complainIfDisposed();

		boolean actionBarsChanged = false;
		ActionContext context = new ActionContext(aStructuredSelection);
		if (isApplyingActionsFromContentExtensions) {
			NavigatorContentDescriptorInstance[] contentDescriptorInstances = contentService.findRelevantContentDescriptorInstances(aStructuredSelection);
			NavigatorContentDescriptorInstance contentDescriptorInstance = null;
			ICommonActionProvider provider = null;
			for (int x = 0; x < contentDescriptorInstances.length; ++x) {
				try {
					contentDescriptorInstance= contentDescriptorInstances[x];
					provider = contentDescriptorInstance.getActionProvider();
					if (provider != SkeletonActionProvider.INSTANCE) {
						provider.setActionContext(context);
						actionBarsChanged |= provider.fillActionBars(theActionBars);
					}
				} catch (RuntimeException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}

		CommonActionProviderDescriptor[] providerDescriptors = findRelevantActionDescriptors(aStructuredSelection);
		if (providerDescriptors.length > 0) {
			ICommonActionProvider provider = null;
			for (int i = 0; i < providerDescriptors.length; i++) {
				try {
					provider = getActionProviderInstance(providerDescriptors[i]);
					provider.setActionContext(context);
					actionBarsChanged |= provider.fillActionBars(theActionBars);
				} catch (RuntimeException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		if (actionBarsChanged) {
			theActionBars.updateActionBars();
			theActionBars.getMenuManager().update();
		}
	}

	private CommonActionProviderDescriptor[] findRelevantActionDescriptors(IStructuredSelection aStructuredSelection) {
		CommonActionProviderDescriptor actionDescriptor = null;
		List providers = new ArrayList();
		for (Iterator providerItr = actionProviderDescriptors.iterator(); providerItr.hasNext();) {
			actionDescriptor = (CommonActionProviderDescriptor) providerItr.next();
			if (actionDescriptor.isEnabledFor(aStructuredSelection))
				providers.add(actionDescriptor);
		}
		if (providers.size() > 0)
			return (CommonActionProviderDescriptor[]) providers.toArray(new CommonActionProviderDescriptor[providers.size()]);
		return NO_DESCRIPTORS;
	}

	public void dispose() {
		if (isDisposed)
			return;
		synchronized (actionProviderInstances) {
			for (Iterator iter = actionProviderInstances.values().iterator(); iter.hasNext();) {
				ICommonActionProvider element = (ICommonActionProvider) iter.next();
				element.dispose();
			}
			actionProviderInstances.clear();
		}
		actionProviderDescriptors.clear();
		isDisposed = true;
	}

	private ICommonActionProvider getActionProviderInstance(CommonActionProviderDescriptor aProviderDescriptor) {
		ICommonActionProvider provider = (ICommonActionProvider) actionProviderInstances.get(aProviderDescriptor);
		if (provider != null)
			return provider;
		synchronized (actionProviderInstances) {
			provider = (ICommonActionProvider) actionProviderInstances.get(aProviderDescriptor);
			if (provider == null) {
				provider = aProviderDescriptor.createActionProvider();
				if (provider != null) {
					initialize(provider);
					actionProviderInstances.put(aProviderDescriptor, provider);
				} else
					actionProviderInstances.put(aProviderDescriptor, (provider = SkeletonActionProvider.INSTANCE));
			}
		}
		return provider;
	}

	private void addActionDescriptor(CommonActionProviderDescriptor aDescriptor) {
		actionProviderDescriptors.add(aDescriptor);
	}

	private void complainIfDisposed() {
		if (isDisposed)
			throw new IllegalStateException("NavigatorActionService has already been disposed.");
	}

	private class CommonActionRegistry extends RegistryReader {

		private static final String ACTION_PROVIDER = "actionProvider"; //$NON-NLS-1$

		public CommonActionRegistry() {
			super(NavigatorPlugin.PLUGIN_ID, ACTION_PROVIDER);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.wst.common.navigator.internal.views.extensions.RegistryReader#readElement(org.eclipse.core.runtime.IConfigurationElement)
		 */
		protected boolean readElement(IConfigurationElement anElement) {
			if (ACTION_PROVIDER.equals(anElement.getName())) {
				addActionDescriptor(new CommonActionProviderDescriptor(anElement));
			}
			return true;
		}
	}

	/**
	 * @param aMemento
	 */
	public void restoreState(IMemento aMemento) {
		memento = aMemento;
		ICommonActionProvider provider = null;
		synchronized (actionProviderInstances) {
			for (Iterator actionProviderIterator = actionProviderInstances.values().iterator(); actionProviderIterator.hasNext();) {
				provider = (ICommonActionProvider) actionProviderIterator.next();
				provider.restoreState(memento);
			}
		}

		Collection contentDescriptors = contentService.getDescriptorInstances();
		if (contentDescriptors.size() > 0) {
			NavigatorContentDescriptorInstance element = null;
			for (Iterator iter = contentDescriptors.iterator(); iter.hasNext();) {
				element = (NavigatorContentDescriptorInstance) iter.next();
				provider = element.getActionProvider();
				provider.restoreState(memento);
			}
		}
	}


	/**
	 * @param memento2
	 */
	public void saveState(IMemento aMemento) {
		memento = aMemento;
		ICommonActionProvider provider = null;
		synchronized (actionProviderInstances) {
			for (Iterator actionProviderIterator = actionProviderInstances.values().iterator(); actionProviderIterator.hasNext();) {
				provider = (ICommonActionProvider) actionProviderIterator.next();
				provider.saveState(memento);
			}
		}

		Collection contentDescriptors = contentService.getDescriptorInstances();
		if (contentDescriptors.size() > 0) {
			NavigatorContentDescriptorInstance element = null;
			for (Iterator iter = contentDescriptors.iterator(); iter.hasNext();) {
				element = (NavigatorContentDescriptorInstance) iter.next();
				provider = element.getActionProvider();
				provider.saveState(memento);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentServiceListener#onLoad(org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentDescriptorInstance)
	 */
	public void onLoad(NavigatorContentDescriptorInstance aDescriptorInstance) {
		initialize(aDescriptorInstance.getActionProvider());
	}

	private void initialize(ICommonActionProvider anActionProvider) {
		if (anActionProvider != null && anActionProvider != SkeletonActionProvider.INSTANCE) {
			anActionProvider.init(viewPart, structuredViewer, contentService);
			if (memento != null)
				anActionProvider.restoreState(memento);
			anActionProvider.setActionContext(new ActionContext(StructuredSelection.EMPTY));
			anActionProvider.fillActionBars(viewPart.getViewSite().getActionBars());
		}
		
	}
}
