/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.navigator.internal.views.extensions;

import java.util.Comparator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IOpenListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonActionProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonLabelProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonOpenListener;
import org.eclipse.wst.common.navigator.internal.provisional.views.INavigatorContentExtension;
import org.eclipse.wst.common.navigator.internal.provisional.views.INavigatorContentProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.NavigatorContentService;

/**
 * <p>
 * The following class is experimental until fully documented.
 * </p>
 */
public class NavigatorContentExtension implements INavigatorContentExtension {

	private NavigatorContentDescriptor descriptor;
	private INavigatorContentProvider contentProvider;
	private ICommonLabelProvider labelProvider;
	private ICommonActionProvider actionProvider;
	private NavigatorContentService contentService;
	private ICommonOpenListener openListener;
	private Comparator comparator;
	private boolean labelProviderInitializationFailed = false;
	private boolean contentProviderInitializationFailed = false;
	private boolean actionProviderInitializationFailed = false;
	private boolean openServiceInitializationFailed = false;
	private boolean comparatorInitializationFailed = false;


	/**
	 * 
	 */
	public NavigatorContentExtension() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#getContentProvider()
	 */
	public INavigatorContentProvider getContentProvider() {
		if (contentProvider != null || contentProviderInitializationFailed)
			return contentProvider;
		synchronized (this) {
			try {
				if (contentProvider == null) {
					ITreeContentProvider treeContentProvider = (ITreeContentProvider) descriptor.getConfigurationElement().createExecutableExtension(NavigatorContentDescriptor.ATT_CONTENT_PROVIDER);
					if (treeContentProvider != null)
						contentProvider = new NavigatorContentProvider(treeContentProvider);
					else
						contentProvider = new NavigatorContentProvider(SkeletonTreeContentProvider.INSTANCE);

				}
			} catch (CoreException e) {
				contentProviderInitializationFailed = true;
				e.printStackTrace();
			} catch (RuntimeException e) {
				contentProviderInitializationFailed = true;
				e.printStackTrace();
			}
			if (contentProviderInitializationFailed)
				contentProvider = new NavigatorContentProvider(SkeletonTreeContentProvider.INSTANCE);
		}
		return contentProvider;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#getLabelProvider()
	 */
	public ICommonLabelProvider getLabelProvider() {
		if (labelProvider != null || labelProviderInitializationFailed)
			return labelProvider;
		synchronized (this) {
			try {

				if (labelProvider == null) {
					ILabelProvider tempLabelProvider = (ILabelProvider) descriptor.getConfigurationElement().createExecutableExtension(NavigatorContentDescriptor.ATT_LABEL_PROVIDER);

					if (tempLabelProvider instanceof ICommonLabelProvider) {
						labelProvider = (ICommonLabelProvider) tempLabelProvider;
						labelProvider.initialize(contentService.getViewerId());
					} else {
						labelProvider = new DelegateCommonLabelProvider(tempLabelProvider);
					}
				}
			} catch (CoreException e) {
				labelProviderInitializationFailed = true;
				e.printStackTrace();
			} catch (RuntimeException e) {
				labelProviderInitializationFailed = true;
				e.printStackTrace();
			}

			if (labelProviderInitializationFailed)
				labelProvider = SkeletonLabelProvider.INSTANCE;
		}
		return labelProvider;
	}

	/**
	 * @return
	 */
	public ICommonActionProvider getActionProvider() {
		if (actionProvider != null || actionProviderInitializationFailed)
			return actionProvider;
		if (descriptor.getConfigurationElement().getAttribute(NavigatorContentDescriptor.ATT_ACTION_PROVIDER) == null) {
			actionProvider = SkeletonActionProvider.INSTANCE;
			return actionProvider;
		}
		synchronized (this) {
			try {
				if (actionProvider == null) {
					actionProvider = (ICommonActionProvider) descriptor.getConfigurationElement().createExecutableExtension(NavigatorContentDescriptor.ATT_ACTION_PROVIDER);
					if (actionProvider == null)
						actionProvider = SkeletonActionProvider.INSTANCE;
				}
			} catch (CoreException e) {
				actionProviderInitializationFailed = true;
				e.printStackTrace();
			} catch (RuntimeException e) {
				actionProviderInitializationFailed = true;
				e.printStackTrace();
			}
			if (actionProviderInitializationFailed)
				actionProvider = SkeletonActionProvider.INSTANCE;
		}
		return actionProvider;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#getContentProvider()
	 */
	public ICommonOpenListener getOpenListener() {
		if (openListener != null || openServiceInitializationFailed)
			return openListener;
		if (descriptor.getConfigurationElement().getAttribute(NavigatorContentDescriptor.ATT_OPEN_LISTENER) == null) {
			openListener = SkeletonOpenListener.INSTANCE;
			return openListener;
		}
		synchronized (this) {
			try {
				if (openListener == null) {
					IOpenListener tempOpenListener = (IOpenListener) descriptor.getConfigurationElement().createExecutableExtension(NavigatorContentDescriptor.ATT_OPEN_LISTENER);

					if (tempOpenListener instanceof ICommonOpenListener) {
						openListener = (ICommonOpenListener) tempOpenListener;
					} else {
						openListener = new DelegateCommonOpenListener(tempOpenListener);
					}
				}
			} catch (CoreException e) {
				openServiceInitializationFailed = true;
				e.printStackTrace();
			} catch (RuntimeException e) {
				openServiceInitializationFailed = true;
				e.printStackTrace();
			}
			if (openServiceInitializationFailed)
				openListener = SkeletonOpenListener.INSTANCE;
		}
		return openListener;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#init(java.lang.String,
	 *      org.eclipse.wst.common.navigator.views.INavigatorExtensionSite)
	 */
	public void init(NavigatorContentDescriptor aContentDescriptor, NavigatorContentService aContentService) {
		descriptor = aContentDescriptor;
		contentService = aContentService;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#dispose()
	 */
	public void dispose() {
		if (contentProvider != null)
			contentProvider.dispose();
		if (labelProvider != null)
			labelProvider.dispose();
		if (actionProvider != null)
			actionProvider.dispose();
	}


	// M4 Revisit the sorting strategy [CommonNavigator:SORTING]
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.views.INavigatorContentExtension#getComparator()
	 */
	public Comparator getComparator() {
		if (comparator != null || comparatorInitializationFailed)
			return comparator;
		synchronized (this) {
			try {
				if (comparator == null) {
					String sorterClassName = descriptor.getConfigurationElement().getAttribute(NavigatorContentDescriptor.ATT_SORTER);
					if (sorterClassName != null && sorterClassName.length() > 0)
						comparator = (Comparator) descriptor.getConfigurationElement().createExecutableExtension(NavigatorContentDescriptor.ATT_SORTER);
				}

			} catch (CoreException e) {
				comparatorInitializationFailed = true;
				e.printStackTrace();
			} catch (RuntimeException e) {
				comparatorInitializationFailed = true;
				e.printStackTrace();
			}
			if (comparatorInitializationFailed || comparator == null)
				comparator = IdentityComparator.INSTANCE;
		}
		return comparator;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		return null;
	}

	/**
	 * @return Returns the contentProviderInitializationFailed.
	 */
	public boolean hasContentProviderInitializationFailed() {
		return contentProviderInitializationFailed;
	}

	/**
	 * @return Returns the labelProviderInitializationFailed.
	 */
	public boolean hasLabelProviderInitializationFailed() {
		return labelProviderInitializationFailed;
	}


}
