 /*******************************************************************************
  * Copyright (c) 2005 Versant Corporation and others.
  * All rights reserved. This program and the accompanying materials
  * are made available under the terms of the Eclipse Public License v1.0
  * which accompanies this distribution, and is available at
  * http://www.eclipse.org/legal/epl-v10.html
  * 
  * Contributors:
  *     Versant Corporation - initial API and implementation
  *******************************************************************************/
 package org.eclipse.wst.rdb.internal.mysql.ddl;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOption;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategory;
import org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlGenerationOptions;
import org.eclipse.wst.rdb.internal.core.containment.ContainmentServiceImpl;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.tables.PersistentTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.ViewTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Trigger;
import org.eclipse.wst.rdb.internal.models.sql.constraints.*;

import java.util.*;

 /**
  * This class uses the MySqlDdlBuilder to generates the sql scripts.
  */
 public class MySqlDdlGenerator implements DDLGenerator {
     private EngineeringOption[] options = null;
     private EngineeringOptionCategory[] categories = null;
     private MySqlDdlBuilder builder = null;

     public String[] generateDDL(SQLObject[] elements,
                                 IProgressMonitor progressMonitor) {
         String[] statements = new String[0];
         if (this.generateCreateStatement()) {
             statements = this.createSQLObjects(elements, this
                     .generateQuotedIdentifiers(), this
                     .generateFullyQualifiedNames(), progressMonitor);
         }
         if (this.generateDropStatement()) {
             String[] drop = this.dropSQLObjects(elements, this
                     .generateQuotedIdentifiers(), this
                     .generateFullyQualifiedNames(), progressMonitor);
             String[] temp = statements;
             statements = new String[temp.length + drop.length];

             System.arraycopy(drop, 0, statements, 0, drop.length);
             System.arraycopy(temp, 0, statements, drop.length , temp.length);
         }
         return statements;
     }

     public String[] createSQLObjects(SQLObject[] elements,
                                      boolean quoteIdentifiers, boolean qualifyNames,
                                      IProgressMonitor progressMonitor) {
         String[] statements = this.createStatements(elements, quoteIdentifiers,
                 qualifyNames, progressMonitor, 100);
         return statements;
     }

     public String[] dropSQLObjects(SQLObject[] elements,
                                    boolean quoteIdentifiers, boolean qualifyNames,
                                    IProgressMonitor progressMonitor) {
         String[] statements = this.dropStatements(elements, quoteIdentifiers,
                 qualifyNames, progressMonitor, 100);
         return statements;
     }

     private String[] createStatements(SQLObject[] elements,
                                       boolean quoteIdentifiers, boolean qualifyNames,
                                       IProgressMonitor progressMonitor, int task) {
         MySqlDdlScript script = new MySqlDdlScript();
         if (this.builder == null) {
             this.builder = new MySqlDdlBuilder();
         }
         Iterator it = this.getAllContainedDisplayableElementSet(elements)
                 .iterator();
         while (it.hasNext()) {
             Object o = it.next();
             if (o instanceof PersistentTable) {
                 if (!this.generateTables())
                     continue;
                 String statement = builder.createTable((PersistentTable) o,
                         quoteIdentifiers, qualifyNames, generatePkConstraints());
                 if (statement != null)
                     script.addCreateTableStatement(statement);
             } else if (o instanceof ViewTable) {
                 if (!this.generateViews())
                     continue;
                 String statement = builder.createView((ViewTable) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addCreateViewStatement(statement);
             } else if (o instanceof Trigger) {
                 if (!this.generateTriggers())
                     continue;
                 String statement = builder.createTrigger((Trigger) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addCreateTriggerStatement(statement);
             } else if (o instanceof CheckConstraint) {
                 String statement = builder.addCheckConstraint(
                         (CheckConstraint) o, quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addAlterTableAddConstraintStatement(statement);
             } else if (o instanceof UniqueConstraint) {
                 String statement = null;
                 if (o instanceof PrimaryKey) {
                     if (!this.generatePkConstraints())
                         continue;
                     statement = builder.addUniqueConstraint(
                             (UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                 } else {
                     if (!this.generateIndexes())
                         continue;
                     statement = builder.addUniqueConstraint(
                             (UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                 }
                 if (statement != null)
                     script.addAlterTableAddConstraintStatement(statement);
             } else if (o instanceof ForeignKey) {
                 if (!this.generateFkConstraints())
                     continue;
                 String statement = builder.addForeignKey((ForeignKey) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addAlterTableAddForeignKeyStatement(statement);
             } else if (o instanceof Index) {
                 if (!this.generateIndexes())
                     continue;
                 String statement = builder.createIndex((Index) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addCreateIndexStatement(statement);
             }
         }
         return script.getStatements();
     }

     private String[] dropStatements(SQLObject[] elements,
                                     boolean quoteIdentifiers, boolean qualifyNames,
                                     IProgressMonitor progressMonitor, int task) {
         MySqlDdlScript script = new MySqlDdlScript();
         if (this.builder == null) {
             this.builder = new MySqlDdlBuilder();
         }

         Iterator it = this.getAllContainedDisplayableElementSet(elements)
                 .iterator();
         while (it.hasNext()) {
             Object o = it.next();
             if (o instanceof PersistentTable) {
                 if (!this.generateTables())
                     continue;
                 String statement = builder.dropTable((PersistentTable) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addDropTableStatement(statement);
             } else if (o instanceof ViewTable) {
                 if (!this.generateViews())
                     continue;
                 String statement = builder.dropView((ViewTable) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addDropViewStatement(statement);
             } else if (o instanceof Trigger) {
                 if (!this.generateTriggers())
                     continue;
                 String statement = builder.dropTrigger((Trigger) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addDropTriggerStatement(statement);
             } else if (o instanceof CheckConstraint) {
                 List statements = builder.dropTableConstraint(
                         (CheckConstraint) o, quoteIdentifiers, qualifyNames);
                 if (!statements.isEmpty()) {
                     for (Iterator iter = statements.iterator(); iter.hasNext();) {
                         String statement = (String) iter.next();
                         script.addAlterTableDropConstraintStatement(statement);
                     }
                 }
             } else if (o instanceof UniqueConstraint) {
                 List statements = null;
                 if (o instanceof PrimaryKey) {
                     if (!this.generatePkConstraints())
                         continue;
                     statements = builder.dropTableConstraint(
                             (UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                 } else {
                     if (!this.generateIndexes())
                         continue;
                     statements = builder.dropTableConstraint(
                             (UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                 }
                 if (!statements.isEmpty()) {
                     for (Iterator iter = statements.iterator(); iter.hasNext();) {
                         String statement = (String) iter.next();
                         script.addAlterTableDropConstraintStatement(statement);
                     }
                 }
             } else if (o instanceof ForeignKey) {
                 if (!this.generateFkConstraints())
                     continue;
                 List statements = builder.dropTableConstraint((ForeignKey) o,
                         quoteIdentifiers, qualifyNames);
                 if (!statements.isEmpty()) {
                     for (Iterator iter = statements.iterator(); iter.hasNext();) {
                         String statement = (String) iter.next();
                         script.addAlterTableDropForeignKeyStatement(statement);
                     }
                 }
             } else if (o instanceof Index) {
                 if (!this.generateIndexes())
                     continue;
                 String statement = builder.dropIndex((Index) o,
                         quoteIdentifiers, qualifyNames);
                 if (statement != null)
                     script.addDropIndexStatement(statement);
             }
         }
         return script.getStatements();
     }

     public EngineeringOption[] getOptions() {
         if(this.options == null) {
             this.options = GenericDdlGenerationOptions.createDDLGenerationOptions(this.getOptionCategories());
         }

         return this.options;
     }

     public EngineeringOptionCategory[] getOptionCategories() {
         if(this.categories == null) {
             this.categories = GenericDdlGenerationOptions.createDDLGenerationOptionCategories();
         }
         return this.categories;
     }

     private boolean generateCreateStatement() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_CREATE_STATEMENTS]
                 .getBoolean();
     }

     private boolean generateDropStatement() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_DROP_STATEMENTS]
                 .getBoolean();
     }

     private boolean generatePkConstraints() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_PK_CONSTRAINTS]
                                  .getBoolean();
     }

     public final boolean generateFkConstraints() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_FK_CONSTRAINTS]
                 .getBoolean();
     }

     private final boolean generateQuotedIdentifiers() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_QUOTED_IDENTIFIER]
                 .getBoolean();
     }

     private final boolean generateFullyQualifiedNames() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_FULLY_QUALIFIED_NAME]
                 .getBoolean();
     }

     public final boolean generateTables() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TABLES]
                 .getBoolean();
     }

     public boolean generateIndexes() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_INDICES]
                 .getBoolean();
     }

     public boolean generateViews() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_VIEWS]
                 .getBoolean();
     }

     public boolean generateTriggers() {
         return this.getOptions()[GenericDdlGenerationOptions.GENERATE_TRIGGERS]
                 .getBoolean();
     }

     private List getAllContainedDisplayableElementSet(SQLObject[] elements) {
         Set s = new HashSet();
         for (int i = 0; i < elements.length; ++i) {
             s.add(elements[i]);
             s.addAll(ContainmentServiceImpl.INSTANCE
                     .getAllContainedDisplayableElements(elements[i]));
         }
         return orderElementSet(s);
     }

     private List orderElementSet(Set unordered) {
         List order = new ArrayList();
         for (Iterator iterator = unordered.iterator(); iterator.hasNext();) {
             Object o = iterator.next();
             if (o instanceof PersistentTable) {
                 order.add(o);
                 iterator.remove();
             }
         }
         order.addAll(unordered);
         return order;
     }

 }
