/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
/*
 * Created on Feb 5, 2004
 *  
 */
package org.eclipse.jst.common.navigator.internal.actions;

import java.util.Iterator;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.ui.IContextMenuConstants;
import org.eclipse.jdt.ui.actions.RefreshAction;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.BuildAction;
import org.eclipse.ui.actions.CloseResourceAction;
import org.eclipse.ui.actions.OpenResourceAction;
import org.eclipse.ui.ide.IDEActionFactory;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonActionProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.NavigatorContentService;
import org.eclipse.wst.common.navigator.internal.views.actions.CommonActionProvider;

/**
 * @author mdelder
 *  
 */
public class ProjectMgmtActionsExtension extends CommonActionProvider implements ICommonActionProvider {
 
	private OpenResourceAction openProjectAction;

	private CloseResourceAction closeProjectAction;

	private BuildAction buildAction;

	private BuildAction rebuildAction;

	private RefreshAction refreshAction;

	private ActionContext actionContext;

	public ProjectMgmtActionsExtension() {
		super();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.wst.common.navigator.internal.views.actions.ICommonActionProvider#init(org.eclipse.ui.IViewPart, org.eclipse.jface.viewers.StructuredViewer, org.eclipse.wst.common.navigator.internal.views.extensions.NavigatorContentService)
	 */
	public void init(IViewPart aViewPart, StructuredViewer aViewer, NavigatorContentService aContentService) {		
		
		Shell shell = (aViewPart.getViewSite() != null) ? aViewPart.getViewSite().getShell() : null;
		openProjectAction = new OpenResourceAction(shell);
		closeProjectAction = new CloseResourceAction(shell);
		buildAction = new BuildAction(shell, IncrementalProjectBuilder.INCREMENTAL_BUILD);
		buildAction.setText(ActionMessages.BuildAction_label); 

		rebuildAction = new BuildAction(shell, IncrementalProjectBuilder.FULL_BUILD);
		rebuildAction.setText(ActionMessages.RebuildAction_label); 

		refreshAction = new RefreshAction(aViewPart.getViewSite());

	}

	/**
	 * Extends the superclass implementation to dispose the subgroups.
	 */
	public void dispose() {
		//Default nothing
	}


	public boolean fillActionBars(IActionBars actionBars) {
		if (actionBars != null) {
			IStructuredSelection selection = getStructureSelection();
			setSelection(selection);
			actionBars.setGlobalActionHandler(ActionFactory.REFRESH.getId(), refreshAction);
			actionBars.setGlobalActionHandler(IDEActionFactory.BUILD_PROJECT.getId(), buildAction);
			actionBars.setGlobalActionHandler(IDEActionFactory.REBUILD_PROJECT.getId(), rebuildAction);
			actionBars.setGlobalActionHandler(IDEActionFactory.OPEN_PROJECT.getId(), openProjectAction);
			actionBars.setGlobalActionHandler(IDEActionFactory.CLOSE_PROJECT.getId(), closeProjectAction);

		}
		return true;
	}

	/**
	 * @param selection
	 */
	private void setSelection(IStructuredSelection selection) {
		if (selection != null) {
			refreshAction.selectionChanged(selection);
			buildAction.selectionChanged(selection);
			rebuildAction.selectionChanged(selection);
			openProjectAction.selectionChanged(selection);
			closeProjectAction.selectionChanged(selection);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.actions.ICommonActionProvider#setActionContext(org.eclipse.ui.actions.ActionContext)
	 */
	public void setActionContext(ActionContext aContext) {
		actionContext = aContext;
	}

	private IStructuredSelection getStructureSelection() {
		if (actionContext != null && !actionContext.getSelection().isEmpty() && (actionContext.getSelection() instanceof IStructuredSelection)) {

			return (IStructuredSelection) actionContext.getSelection();
		}
		return null;

	}

	public boolean fillContextMenu(IMenuManager aMenu) {
		IStructuredSelection selection = getStructureSelection();
		setSelection(selection);
		fillContextMenu(aMenu, selection);
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.navigator.INavigatorActionsExtension#fillContextMenu(org.eclipse.jface.action.IMenuManager,
	 *      org.eclipse.jface.viewers.IStructuredSelection)
	 */
	private void fillContextMenu(IMenuManager menu, IStructuredSelection selection) {
		boolean isProjectSelection = true;
		boolean hasOpenProjects = false;
		boolean hasClosedProjects = false;
		boolean hasBuilder = true; // false if any project is closed or does not
		// have builder
		Iterator resources = selection.iterator();

		while (resources.hasNext() && (!hasOpenProjects || !hasClosedProjects || hasBuilder || isProjectSelection)) {
			Object next = resources.next();
			IProject project = null;

			if (next instanceof IProject)
				project = (IProject) next;
			else if (next instanceof IAdaptable)
				project = (IProject) ((IAdaptable) next).getAdapter(IProject.class);

			if (project == null) {
				isProjectSelection = false;
				continue;
			}
			if (project.isOpen()) {
				hasOpenProjects = true;
				if (hasBuilder && !hasBuilder(project))
					hasBuilder = false;
			} else {
				hasClosedProjects = true;
				hasBuilder = false;
			}
		}
		if (!selection.isEmpty() && isProjectSelection && !ResourcesPlugin.getWorkspace().isAutoBuilding() && hasBuilder) {
			// Allow manual incremental build only if auto build is off.
			appendToGroup(menu, buildAction);
		}
		if (!hasClosedProjects) {
			appendToGroup(menu, refreshAction);
		}
		if (isProjectSelection) {
			if (hasClosedProjects)
				appendToGroup(menu, openProjectAction);
			if (hasOpenProjects)
				appendToGroup(menu, closeProjectAction);
		}
	}

	/**
	 * Returns whether there are builders configured on the given project.
	 * 
	 * @return <code>true</code> if it has builders, <code>false</code> if not, or if this could
	 *         not be determined
	 */
	private boolean hasBuilder(IProject project) {
		try {
			ICommand[] commands = project.getDescription().getBuildSpec();
			if (commands.length > 0)
				return true;
		} catch (CoreException e) {
			// Cannot determine if project has builders. Project is closed
			// or does not exist. Fall through to return false.
		}
		return false;
	}


	protected void appendToGroup(IMenuManager menu, IAction action) {
		if (action.isEnabled())
			menu.appendToGroup(IContextMenuConstants.GROUP_BUILD, action);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter) {
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}
}
