/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.core;

import java.util.Iterator;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.wst.rdb.data.internal.core.common.DefaultColumnDataAccessor;
import org.eclipse.wst.rdb.data.internal.core.common.IColumnDataAccessor;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinitionRegistry;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.UserDefinedType;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;
import org.osgi.framework.BundleContext;


/**
 * The main plugin class to be used in the desktop.
 */
public class DataCorePlugin extends Plugin
{
	//The shared instance.
	private static DataCorePlugin plugin;
	//Resource bundle.
	private ResourceBundle resourceBundle;
    private static final String SQ = "'"; //$NON-NLS-1$ //$NON-NLS-2$
    private static final String SQ2 = "`"; //$NON-NLS-1$ //$NON-NLS-2$
    private static final String DQ = "\""; //$NON-NLS-1$ //$NON-NLS-2$

	public static final String ID = "org.eclipse.wst.rdb.data.core"; //$NON-NLS-1$
	
	protected Vector columnDataAccessors;
	
	/**
	 * The constructor.
	 */
	public DataCorePlugin() {
		super();
		plugin = this;
		try {
			resourceBundle = ResourceBundle.getBundle("org.eclipse.wst.rdb.data.internal.core.CorePluginResources"); //$NON-NLS-1$			
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
		listColumnDataAccessors();
	}

	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}

	/**
	 * Returns the shared instance.
	 */
	public static DataCorePlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle = DataCorePlugin.getDefault().getResourceBundle();
		try {
			return (bundle != null) ? bundle.getString(key) : key;
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}
	
	/**
	 * Method writeLog.
	 * @param severity - the severity; one of IStatus.OK, IStatus.ERROR, IStatus.INFO, or IStatus.WARNING
	 * @param code - the plug-in-specific status code, or OK
	 * @param message - a human-readable message, localized to the current locale
	 * @param exception- a low-level exception, or null if not applicable
	 */
	public void writeLog(int severity, int code, String message, Throwable exception) {
		if (message == null)
			message = ""; //$NON-NLS-1$
		getLog().log(
				new Status(severity, getBundle().getSymbolicName(), code, message, exception));
	}
	
    public static String getQualifiedTableName(Table table) {        
        return quoteIdentifier(table.getSchema().getDatabase(), table.getSchema().getName()) 
            + "." + quoteIdentifier(table.getSchema().getDatabase(), table.getName()); //$NON-NLS-1$
    }
    
    public static String getQualifiedUDTName(UserDefinedType udt)
    {
        return quoteIdentifier(udt.getSchema().getDatabase(), udt.getSchema().getName()) 
        + "." + quoteIdentifier(udt.getSchema().getDatabase(), udt.getName()); //$NON-NLS-1$
    }
    
    public static String quoteIdentifier(Database db, String s)
    {
//        String delim = RDBCorePlugin.getDefault().getConnectionManager().getConnectionInfo(o).getIdentifierQuoteString();
//        if (delim == null || delim.length() == 0)
//            delim = "\""; //$NON-NLS-1$
    	String delim = DQ;
		DatabaseDefinitionRegistry dbDefRegistry = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry();
		DatabaseDefinition dbDef =  dbDefRegistry.getDefinition(db);
		delim = dbDef.getIdentifierQuoteString();
        if (!SQ.equals(delim) && !delim.equals(SQ2)){
            delim = DQ; 
        }
        return delim + doubleStringDelim(s, delim) + delim;
    }
    
    public static String doubleStringDelim(String s, String delim)
	{
        if (delim==null||delim.length()==0)
            return s;
        
        int i = s.indexOf(delim);
        if(i != -1) {
            StringBuffer sb = new StringBuffer(s);            
            int j;
            for(; i != -1; i = sb.toString().indexOf(delim, j)) {
                sb = sb.insert(i, delim);
                j = i + 2*delim.length();
            }
            return sb.toString();
        } else
            return s;
	}
    
    protected void listColumnDataAccessors()
    {
    	columnDataAccessors = new Vector();
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = registry.getExtensionPoint(ID + ".columnDataAccessors"); //$NON-NLS-1$
		IExtension[] extensions = extensionPoint.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IExtension extension = extensions[i];
			IConfigurationElement[] elements = extension.getConfigurationElements();
			for (int j=0; j<elements.length; ++j)
				columnDataAccessors.add(new ColumnDataAccessorExtension(elements[j]));
		}
	}
    
    public IColumnDataAccessor newColumnDataAccessor(Column sqlCol)
    {
    	IColumnDataAccessor acc;
    	try {
	    	Database db = sqlCol.getTable().getSchema().getDatabase();
	    	String vendor = db.getVendor();
	    	String version = db.getVersion();
	    	String dataType = sqlCol.getDataType().getName();
	    	
	    	ColumnDataAccessorExtension element = null;
	    	int score = -1;
	    	
	    	Iterator it = columnDataAccessors.iterator();
	    	while (it.hasNext()) {
	    		ColumnDataAccessorExtension curElement = (ColumnDataAccessorExtension)it.next();
				if ( curElement.matches(vendor, version, dataType) && 
					  (element==null || curElement.getScore()>element.getScore()) ) {
					element = curElement;
					score = curElement.getScore();
				}
	    	}
	    	
	    	if (element!=null)
	    		acc = element.createInstance();
	    	else
	    		acc = new DefaultColumnDataAccessor();
	    	
    	} catch (CoreException ex) {
    		writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
    		acc = new DefaultColumnDataAccessor();
    	}
    	acc.initialize(sqlCol);
		return acc;
    }
    
}

class ColumnDataAccessorExtension
{	
	protected IConfigurationElement element;
	protected String vendors;
	protected String versions;
	protected String dataTypes;
	protected int score;
	
	public ColumnDataAccessorExtension(IConfigurationElement element)
	{
		this.element = element;
		vendors = element.getAttribute("vendor"); //$NON-NLS-1$
		versions = element.getAttribute("version"); //$NON-NLS-1$
		dataTypes = element.getAttribute("dataType"); //$NON-NLS-1$
		score = ((vendors!=null)?1:0) + ((versions!=null)?1:0) + ((dataTypes!=null)?1:0);
	}
	
	public boolean matches(String vendor, String version, String dataType)
	{
		if (this.vendors!=null && !contains(this.vendors, vendor))
			return false;
		else if (this.versions!=null && !contains(this.versions,version))
			return false;
		else if (this.dataTypes!=null && !contains(this.dataTypes,dataType))
			return false;
		else
			return true;
	}
	
	protected boolean contains(String values, String value)
	{
		StringTokenizer st = new StringTokenizer(values, ",", false); //$NON-NLS-1$
		while (st.hasMoreElements()) {
			String s = st.nextToken();
			if (value.equals(s.trim()))
				return true;
		}
		return false;
	}
	
	public IColumnDataAccessor createInstance() throws CoreException
	{
		return (IColumnDataAccessor)element.createExecutableExtension("class"); //$NON-NLS-1$
	}
	
	public int getScore()
	{
		return score;
	}
}
