/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.common.navigator.internal.actions;

import java.lang.reflect.InvocationTargetException;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.corext.refactoring.reorg.JavaCopyProcessor;
import org.eclipse.jdt.internal.corext.refactoring.reorg.JavaMoveProcessor;
import org.eclipse.jdt.internal.corext.refactoring.reorg.ReorgUtils;
import org.eclipse.jdt.internal.ui.packageview.PackagesMessages;
import org.eclipse.jdt.internal.ui.refactoring.RefactoringMessages;
import org.eclipse.jdt.internal.ui.refactoring.reorg.ReorgCopyStarter;
import org.eclipse.jdt.internal.ui.refactoring.reorg.ReorgMoveStarter;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.ui.actions.CopyFilesAndFoldersOperation;
import org.eclipse.ui.views.navigator.LocalSelectionTransfer;
import org.eclipse.wst.common.navigator.internal.views.dnd.CommonNavigatorDropAdapter;
import org.eclipse.wst.common.navigator.internal.views.dnd.IDropValidator;
import org.eclipse.wst.common.navigator.internal.views.dnd.NavigatorDropActionDelegate;


/**
 * @author mdelder Derived from SelectionTransferDropAdapter in JDT internals.
 */
public class JavaDropAction extends NavigatorDropActionDelegate implements IDropValidator {

	private List fElements;
	private JavaMoveProcessor fMoveProcessor;
	private int fCanMoveElements;
	private JavaCopyProcessor fCopyProcessor;
	private int fCanCopyElements;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.navigator.dnd.NavigatorDropActionDelegate#run(org.eclipse.wst.common.navigator.internal.views.navigator.dnd.CommonNavigatorDropAdapter,
	 *      java.lang.Object, java.lang.Object)
	 */
	public boolean run(CommonNavigatorDropAdapter dropAdapter, Object source, Object target) {
		if (LocalSelectionTransfer.getInstance().isSupportedType(dropAdapter.getCurrentTransfer())) {
			try {

				switch (dropAdapter.getCurrentOperation()) {
					case DND.DROP_MOVE :
						handleDropMove(target);
						break;
					case DND.DROP_COPY :
						handleDropCopy(target);
						break;
				}
			} catch (JavaModelException e) {
				ExceptionHandler.handle(e, PackagesMessages.SelectionTransferDropAdapter_error_title, PackagesMessages.SelectionTransferDropAdapter_error_message); 
			} catch (InvocationTargetException e) {
				ExceptionHandler.handle(e, RefactoringMessages.OpenRefactoringWizardAction_refactoring, RefactoringMessages.OpenRefactoringWizardAction_exception); 
			} catch (InterruptedException e) {
				//ok
			} finally {
				// The drag source listener must not perform any operation
				// since this drop adapter did the remove of the source even
				// if we moved something.
				//event.detail= DND.DROP_NONE;
			}
			clear();
			return true;
		} else if (FileTransfer.getInstance().isSupportedType(dropAdapter.getCurrentTransfer())) {
			try {

				final Object data = FileTransfer.getInstance().nativeToJava(dropAdapter.getCurrentTransfer());
				if (data == null || !(data instanceof String[]))
					return false;

				final IContainer targetContainer = getActualTarget(target);
				if (targetContainer == null)
					return false;


				getShell().forceActive();
				new CopyFilesAndFoldersOperation(getShell()).copyFiles((String[]) data, targetContainer);
			} catch (JavaModelException e) {
				String title = PackagesMessages.DropAdapter_errorTitle; 
				String message = PackagesMessages.DropAdapter_errorMessage; 
				ExceptionHandler.handle(e, getShell(), title, message);
			}
			return true;
		}
		return false;
	}

	private IContainer getActualTarget(Object dropTarget) throws JavaModelException {
		if (dropTarget instanceof IContainer)
			return (IContainer) dropTarget;
		else if (dropTarget instanceof IJavaElement)
			return getActualTarget(((IJavaElement) dropTarget).getCorrespondingResource());
		return null;
	}

	protected void initializeSelection() {
		if (fElements != null)
			return;
		ISelection s = LocalSelectionTransfer.getInstance().getSelection();
		if (!(s instanceof IStructuredSelection))
			return;
		fElements = ((IStructuredSelection) s).toList();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.wst.common.navigator.internal.views.navigator.dnd.IDropValidator#validateDrop(org.eclipse.wst.common.navigator.internal.views.navigator.dnd.CommonNavigatorDropAdapter,
	 *      java.lang.Object, int, org.eclipse.swt.dnd.TransferData)
	 */
	public boolean validateDrop(CommonNavigatorDropAdapter dropAdapter, Object target, int operation, TransferData transferType) {

		boolean result = true;
		if (LocalSelectionTransfer.getInstance().isSupportedType(transferType)) {
			initializeSelection();
			try {
				switch (operation) {
					case DND.DROP_DEFAULT :
						result = handleValidateDefault(target) != DND.DROP_NONE;
						break;
					case DND.DROP_COPY :
						result = handleValidateCopy(target) != DND.DROP_NONE;
						break;
					case DND.DROP_MOVE :
						result = handleValidateMove(target) != DND.DROP_NONE; 
						break;
				}
			} catch (JavaModelException e) {
				ExceptionHandler.handle(e, PackagesMessages.SelectionTransferDropAdapter_error_title, PackagesMessages.SelectionTransferDropAdapter_error_message); 
				//event.detail= DND.DROP_NONE;
				result = false;
			}
		}
		return result;
	}


 


	private void handleDropMove(final Object target) throws JavaModelException, InvocationTargetException, InterruptedException {
		IJavaElement[] javaElements = ReorgUtils.getJavaElements(fElements);
		IResource[] resources = ReorgUtils.getResources(fElements);
		ReorgMoveStarter starter = null;
		if (target instanceof IResource)
			starter = ReorgMoveStarter.create(javaElements, resources, (IResource) target);
		else if (target instanceof IJavaElement)
			starter = ReorgMoveStarter.create(javaElements, resources, (IJavaElement) target);
		if (starter != null)
			starter.run(getShell());
	}


	private void handleDropCopy(final Object target) throws JavaModelException, InvocationTargetException, InterruptedException {
		IJavaElement[] javaElements = ReorgUtils.getJavaElements(fElements);
		IResource[] resources = ReorgUtils.getResources(fElements);
		ReorgCopyStarter starter = null;
		if (target instanceof IResource)
			starter = ReorgCopyStarter.create(javaElements, resources, (IResource) target);
		else if (target instanceof IJavaElement)
			starter = ReorgCopyStarter.create(javaElements, resources, (IJavaElement) target);
		if (starter != null)
			starter.run(getShell());
	}



	private int handleValidateCopy(Object target) throws JavaModelException {

		if (fCopyProcessor == null)
			fCopyProcessor= JavaCopyProcessor.create(ReorgUtils.getResources(fElements), ReorgUtils.getJavaElements(fElements));

		if (!canCopyElements())
			return DND.DROP_NONE;	

		if (target instanceof IResource && fCopyProcessor != null && fCopyProcessor.setDestination((IResource)target).isOK())
			return DND.DROP_COPY;
		else if (target instanceof IJavaElement && fCopyProcessor != null && fCopyProcessor.setDestination((IJavaElement)target).isOK())
			return DND.DROP_COPY;
		else
			return DND.DROP_NONE;					
	}


	private int handleValidateDefault(Object target) throws JavaModelException {
		if (target == null)
			return DND.DROP_NONE;

		return handleValidateMove(target);
	}

	private int handleValidateMove(Object target) throws JavaModelException {
		if (target == null)
			return DND.DROP_NONE;
		
		if (fMoveProcessor == null)
			fMoveProcessor= JavaMoveProcessor.create(ReorgUtils.getResources(fElements), ReorgUtils.getJavaElements(fElements));

		if (!canMoveElements())
			return DND.DROP_NONE;	

		if (target instanceof IResource && fMoveProcessor != null && fMoveProcessor.setDestination((IResource)target).isOK())
			return DND.DROP_MOVE;
		else if (target instanceof IJavaElement && fMoveProcessor != null && fMoveProcessor.setDestination((IJavaElement)target).isOK())
			return DND.DROP_MOVE;
		else
			return DND.DROP_NONE;	
	}


	private boolean canMoveElements() {
		if (fCanMoveElements == 0) {
			fCanMoveElements = 2;
			if (fMoveProcessor == null)
				fCanMoveElements = 1;
		}
		return fCanMoveElements == 2;
	}


	private boolean canCopyElements() {
		if (fCanCopyElements == 0) {
			fCanCopyElements = 2;
			if (fCopyProcessor == null)
				fCanCopyElements = 1;
		}
		return fCanCopyElements == 2;
	}

	private void clear() {
		fElements = null;
		fMoveProcessor = null;
		fCanMoveElements = 0;
		fCopyProcessor = null;
		fCanCopyElements = 0;
	}

}
