/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jst.common.navigator.internal.ui.workingsets;

import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.actions.ActionGroup;

/**
 * An action group to provide access to the working sets.
 */
public class ViewActionGroup extends ActionGroup {

	public static final int SHOW_PROJECTS= 1;
	public static final int SHOW_WORKING_SETS= 2;
	public static final String MODE_CHANGED= ViewActionGroup.class.getName() + ".mode_changed"; //$NON-NLS-1$
	static final String TAG_ROOT_MODE= "viewActionMode"; //$NON-NLS-1$
	
	private static final Integer INT_SHOW_PROJECTS= new Integer(SHOW_PROJECTS);
	private static final Integer INT_SHOW_WORKING_SETS= new Integer(SHOW_WORKING_SETS);
	
	private IPropertyChangeListener fChangeListener;
	private ViewAction projects;
	private ViewAction workingSets;
	
	private int fMode;
	private IMenuManager fMenuManager;
	private IWorkingSetActionGroup fActiveActionGroup;
	private WorkingSetShowActionGroup fShowActionGroup;
	private WorkingSetFilterActionGroup fFilterActionGroup;
	

	public ViewActionGroup(int mode, IPropertyChangeListener changeListener, Shell shell) {
		fChangeListener= changeListener;
		fFilterActionGroup= new WorkingSetFilterActionGroup(shell, changeListener);
		fShowActionGroup= new WorkingSetShowActionGroup(shell);
		fMode= mode;
		if (showWorkingSets())
			fActiveActionGroup= fShowActionGroup;
		else
			fActiveActionGroup= fFilterActionGroup;
	}
	
	public void dispose() {
		fFilterActionGroup.dispose();
		fShowActionGroup.dispose();
		fChangeListener= null;
		super.dispose();
	}
	
	public void setWorkingSetModel(WorkingSetModel model) {
		fShowActionGroup.setWorkingSetMode(model);
	}

	/**
	 * {@inheritDoc}
	 */
	public void fillActionBars(IActionBars actionBars) {
		super.fillActionBars(actionBars);
		fMenuManager= actionBars.getMenuManager();
		IMenuManager showMenu= new MenuManager(WorkingSetMessages.getString("ViewActionGroup.show.label")); //$NON-NLS-1$
		fillShowMenu(showMenu);
		fMenuManager.add(showMenu);
		fMenuManager.add(new Separator(IWorkingSetActionGroup.ACTION_GROUP));
		if (fActiveActionGroup == null)
			fActiveActionGroup= fFilterActionGroup;
		((ActionGroup)fActiveActionGroup).fillActionBars(actionBars);
	}
	
	private void fillShowMenu(IMenuManager menu) {
		projects= new ViewAction(this, SHOW_PROJECTS);
		projects.setText(WorkingSetMessages.getString("ViewActionGroup.projects.label")); //$NON-NLS-1$
		menu.add(projects);
		workingSets= new ViewAction(this, SHOW_WORKING_SETS);
		workingSets.setText(WorkingSetMessages.getString("ViewActionGroup.workingSets.label")); //$NON-NLS-1$
		menu.add(workingSets);
		if (fMode == SHOW_PROJECTS) {
			projects.setChecked(true);
		} else {
			workingSets.setChecked(true);
		}
	}

	public void fillFilters(StructuredViewer viewer) {
		ViewerFilter workingSetFilter= fFilterActionGroup.getWorkingSetFilter();
		if (showProjects()) {
			viewer.addFilter(workingSetFilter);
		} else if (showWorkingSets()) {
			viewer.removeFilter(workingSetFilter);
		}
	}
	
	public void setMode(int mode) {
		fMode= mode;
		fActiveActionGroup.cleanViewMenu(fMenuManager);
		PropertyChangeEvent event;
		if (mode == SHOW_PROJECTS) {
			fActiveActionGroup= fFilterActionGroup;
			event= new PropertyChangeEvent(this, MODE_CHANGED, INT_SHOW_WORKING_SETS, INT_SHOW_PROJECTS);
		} else {
			fActiveActionGroup= fShowActionGroup;
			event= new PropertyChangeEvent(this, MODE_CHANGED, INT_SHOW_PROJECTS, INT_SHOW_WORKING_SETS);
		}
		fActiveActionGroup.fillViewMenu(fMenuManager);
		if (fMenuManager != null)
			fMenuManager.updateAll(true);
		fChangeListener.propertyChange(event);
		
	}
	
	public WorkingSetFilterActionGroup getFilterGroup() {
		return fFilterActionGroup;
	}

	public void restoreState(IMemento memento) {
		fFilterActionGroup.restoreState(memento);
		restoreRootMode(memento);
		setMode(fMode);
		if (fMode == SHOW_PROJECTS 
				&& projects != null
				&& workingSets != null) {
			projects.setChecked(true);
			workingSets.setChecked(false);
		} else {
			workingSets.setChecked(true);
			projects.setChecked(false);
		}
	}

	public void saveState(IMemento memento) {
		fFilterActionGroup.saveState(memento);
		memento.putInteger(TAG_ROOT_MODE, fMode);
		
	}
	
	private boolean showProjects() {
		return fMode == SHOW_PROJECTS;
	}
	
	private boolean showWorkingSets() {
		return fMode == SHOW_WORKING_SETS;
	}
	
	private void restoreRootMode(IMemento memento) {
		if (memento != null) {
			Integer value= memento.getInteger(TAG_ROOT_MODE);
			fMode= value == null ? ViewActionGroup.SHOW_PROJECTS : value.intValue();
			if (fMode != ViewActionGroup.SHOW_PROJECTS && fMode != ViewActionGroup.SHOW_WORKING_SETS)
				fMode= ViewActionGroup.SHOW_PROJECTS;
		} else {
			fMode= ViewActionGroup.SHOW_PROJECTS;
		}
	}
	
	
}
