/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.navigator.internal.views.extensions;

import java.util.Comparator;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonActionProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonLabelProvider;
import org.eclipse.wst.common.navigator.internal.provisional.views.ICommonOpenListener;
import org.eclipse.wst.common.navigator.internal.provisional.views.INavigatorContentExtension;
import org.eclipse.wst.common.navigator.internal.provisional.views.NavigatorContentService;

/**
 * <p>
 * The following class is experimental until fully documented.
 * </p>
 * <p>
 * This class is a first step towards componentizing the extension point.
 * </p>
 */
public class NavigatorContentDescriptorInstance {

	private final NavigatorContentDescriptor descriptor;
	private final NavigatorContentService contentService;
	private final String viewerId;

	private INavigatorContentExtension contentExtension;
	private ITreeContentProvider contentProvider;
	private ICommonLabelProvider labelProvider;
	private ICommonActionProvider actionProvider;
	private ICommonOpenListener openListener;
	private Comparator sorter;
	private boolean isDisposed = false;

	public NavigatorContentDescriptorInstance(NavigatorContentDescriptor aDescriptor, NavigatorContentService aManager, String aViewerId) {
		super();

		if (aDescriptor == null)
			throw new IllegalArgumentException("The parameter aDescriptor (of type " + NavigatorContentDescriptor.class.getName() + ") cannot be null!");

		descriptor = aDescriptor;
		contentService = aManager;
		viewerId = aViewerId;
	}

	public ITreeContentProvider getContentProvider() {
		complainDisposedIfNecessary();
		if (contentProvider != null)
			return contentProvider;
		synchronized (this) {
			if (contentProvider == null) {
				try {
					contentProvider = getContentExtension().getContentProvider();
					contentService.initialize(contentProvider);
				} catch (RuntimeException re) {
					contentProvider = SkeletonTreeContentProvider.INSTANCE;
					re.printStackTrace();
				}
			}
		}
		return contentProvider;
	}

	public ICommonLabelProvider getLabelProvider() {
		complainDisposedIfNecessary();
		if (labelProvider == null) {
			try {
				ILabelProvider provider = getContentExtension().getLabelProvider();
				if (provider == null)
					labelProvider = SkeletonLabelProvider.INSTANCE;
				else if (provider instanceof ICommonLabelProvider)
					labelProvider = (ICommonLabelProvider) provider;
				else
					labelProvider = new DelegateCommonLabelProvider(provider);
			} catch (RuntimeException re) {
				labelProvider = SkeletonLabelProvider.INSTANCE;
				re.printStackTrace();
			}
		}
		return labelProvider;
	}
	
	
	public ICommonOpenListener getOpenListener() {
		complainDisposedIfNecessary();
		if (openListener == null) {
			try {
				if (getContentExtension() instanceof NavigatorContentExtension) 
					openListener = ((NavigatorContentExtension) getContentExtension()).getOpenListener();
				
			} catch (RuntimeException re) {
				re.printStackTrace();
			}
			if (openListener == null)
				openListener = SkeletonOpenListener.INSTANCE;

		}
		return openListener;
	}


	public ICommonActionProvider getActionProvider() {
		complainDisposedIfNecessary();
		if (actionProvider != null)
			return actionProvider;
		synchronized (this) {
			if (actionProvider == null) {
				if (getContentExtension() instanceof NavigatorContentExtension)
					actionProvider = ((NavigatorContentExtension) getContentExtension()).getActionProvider();
				else
					actionProvider = SkeletonActionProvider.INSTANCE;
			}
		}
		return actionProvider;
	}
	
	//	 M4 The sorting framework needs further definition [CommonNavigator:SORTING] 
	public Comparator getSorter() {
		complainDisposedIfNecessary();
		if (sorter != null)
			return sorter;
		synchronized (this) {
			if (sorter == null) {
				if (getContentExtension() instanceof NavigatorContentExtension)
					sorter = ((NavigatorContentExtension) getContentExtension()).getComparator();
				else 
					sorter = IdentityComparator.INSTANCE;
			}
		}
		return sorter;		
	}

	public void dispose() {
		synchronized (this) {
			getContentExtension().dispose();
			contentExtension = null;
			isDisposed = true;
		}
	}

	public boolean isLoaded() {
		return contentProvider != null;
	}

	/**
	 * @return Returns the contentExtension.
	 */
	public INavigatorContentExtension getContentExtension() {
		if (contentExtension != null)
			return contentExtension;
		synchronized (this) {
			try {
				if (contentExtension == null) {
					contentExtension = descriptor.createExtension(viewerId);
					contentExtension.init(descriptor, contentService);
				}
			} catch (RuntimeException re) {
				re.printStackTrace();
				contentExtension = SkeletonNavigatorContentExtension.INSTANCE;
			}
		}
		return contentExtension;
	}

	public boolean hasLoadingFailed() {
		if (contentExtension == null)
			return false;
		else if (contentExtension instanceof NavigatorContentExtension) {
			NavigatorContentExtension internalExtension = (NavigatorContentExtension) contentExtension;
			return internalExtension.hasContentProviderInitializationFailed() || internalExtension.hasLabelProviderInitializationFailed();
		}
		return false;
	}

	protected final void complainDisposedIfNecessary() {
		if (isDisposed)
			throw new IllegalStateException("NavigatorContentDescriptorInstance \"" + descriptor.getId() + "\" is disposed!");
	}
}
