/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.ui.properties.internal.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.boot.BootLoader;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IPluginPrerequisite;
import org.eclipse.core.runtime.Platform;
import org.eclipse.wst.common.ui.properties.internal.CommonUIPropertiesPlugin;


/**
 * Utility class.
 * 
 * @author Anthony Hunter 
 * <a href="mailto:anthonyh@ca.ibm.com">anthonyh@ca.ibm.com</a>
 */
public final class TabbedPropertyRegistryUtils {

	/**
	 * Given an extension point name returns its configuration elements.
	 */
	public static IConfigurationElement[] getConfigurationElements(String extensionPointId) {
		IExtensionPoint extensionPoint =
			CommonUIPropertiesPlugin
				.getPlugin()
				.getDescriptor()
				.getExtensionPoint(
				extensionPointId);
		if (extensionPoint == null) {
			return null;
		}
		return extensionPoint.getConfigurationElements();
	}

	/**
	 * Returns the plugin descriptors from this collection of IConfigurationElement objects.
	 */
	public static IPluginDescriptor[] getPluginDescriptors(Collection configurationElements) {
		Set result = new HashSet(configurationElements.size());
		for (Iterator iter = configurationElements.iterator();
			iter.hasNext();
			) {
			IConfigurationElement element = (IConfigurationElement) iter.next();
			result.add(
				element.getDeclaringExtension().getDeclaringPluginDescriptor());
		}
		return (IPluginDescriptor[]) result.toArray(
			new IPluginDescriptor[result.size()]);
	}

	/**
	 * Returns an ordered plugin id list based on the given plugin descriptors.
	 * The order is based on plugin prerequisites.
	 */
	public static String[] computePrerequisiteOrder(IPluginDescriptor[] plugins) {
		String[] orderedPlugins = computePrerequisiteOrderPlugins(plugins);
		return orderedPlugins;
	}

	private static String[] computePrerequisiteOrderPlugins(IPluginDescriptor[] plugins) {
		List prereqs = new ArrayList(9);
		Set pluginList = new HashSet(plugins.length);
		for (int i = 0; i < plugins.length; i++)
			pluginList.add(plugins[i].getUniqueIdentifier());

		// create a collection of directed edges from plugin to prereq
		for (int i = 0; i < plugins.length; i++) {
			boolean boot = false;
			boolean runtime = false;
			boolean found = false;
			IPluginPrerequisite[] prereqList =
				plugins[i].getPluginPrerequisites();
			if (prereqList != null) {
				for (int j = 0; j < prereqList.length; j++) {
					// ensure that we only include values from the original set.
					String prereq = prereqList[j].getUniqueIdentifier();
					boot = boot || prereq.equals(BootLoader.PI_BOOT);
					runtime = runtime || prereq.equals(Platform.PI_RUNTIME);
					if (pluginList.contains(prereq)) {
						found = true;
						prereqs.add(
							new String[] {
								plugins[i].getUniqueIdentifier(),
								prereq });
					}
				}
			}

			// if we didn't find any prereqs for this plugin, add a null prereq
			// to ensure the value is in the output	
			if (!found)
				prereqs.add(
					new String[] { plugins[i].getUniqueIdentifier(), null });
		}

		// do a topological sort, insert the fragments into the sorted elements
		String[][] prereqArray =
			(String[][]) prereqs.toArray(new String[prereqs.size()][]);
		return computeNodeOrder(prereqArray);
	}

	private static String[] computeNodeOrder(String[][] specs) {
		Map counts = computeCounts(specs);
		List nodes = new ArrayList(counts.size());
		while (!counts.isEmpty()) {
			List roots = findRootNodes(counts);
			if (roots.isEmpty())
				break;
			for (Iterator i = roots.iterator(); i.hasNext();)
				counts.remove(i.next());
			nodes.addAll(roots);
			removeArcs(specs, roots, counts);
		}
		String[] result = new String[nodes.size()];
		nodes.toArray(result);

		return result;
	}

	private static HashMap computeCounts(String[][] mappings) {
		HashMap counts = new HashMap(5);
		for (int i = 0; i < mappings.length; i++) {
			String from = mappings[i][0];
			Integer fromCount = (Integer) counts.get(from);
			String to = mappings[i][1];
			if (to == null)
				counts.put(from, new Integer(0));
			else {
				if (((Integer) counts.get(to)) == null)
					counts.put(to, new Integer(0));
				fromCount =
					fromCount == null
						? new Integer(1)
						: new Integer(fromCount.intValue() + 1);
				counts.put(from, fromCount);
			}
		}
		return counts;
	}

	private static List findRootNodes(Map counts) {
		List result = new ArrayList(5);
		for (Iterator i = counts.keySet().iterator(); i.hasNext();) {
			String node = (String) i.next();
			int count = ((Integer) counts.get(node)).intValue();
			if (count == 0)
				result.add(node);
		}
		return result;
	}

	private static void removeArcs(
		String[][] mappings,
		List roots,
		Map counts) {
		for (Iterator j = roots.iterator(); j.hasNext();) {
			String root = (String) j.next();
			for (int i = 0; i < mappings.length; i++) {
				if (root.equals(mappings[i][1])) {
					String input = mappings[i][0];
					Integer count = (Integer) counts.get(input);
					if (count != null)
						counts.put(input, new Integer(count.intValue() - 1));
				}
			}
		}
	}

}
