/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.internal.derby.ddl;


import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.wst.rdb.internal.core.rte.DDLGenerator;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOption;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategory;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionCategoryID;
import org.eclipse.wst.rdb.internal.core.rte.EngineeringOptionID;
import org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlGenerationOptions;
import org.eclipse.wst.rdb.internal.models.sql.constraints.CheckConstraint;
import org.eclipse.wst.rdb.internal.models.sql.constraints.ForeignKey;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.constraints.UniqueConstraint;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.UserDefinedType;
import org.eclipse.wst.rdb.internal.models.sql.routines.Procedure;
import org.eclipse.wst.rdb.internal.models.sql.routines.UserDefinedFunction;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.schema.Sequence;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;
import org.eclipse.wst.rdb.internal.models.sql.tables.Trigger;
import org.eclipse.wst.rdb.internal.models.sql.tables.ViewTable;

public final class DerbyDdlGenerator implements DDLGenerator ,IExecutableExtension {
	
	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
		this.product = config.getAttribute("product"); //$NON-NLS-1$
		this.version = config.getAttribute("version"); //$NON-NLS-1$
	}
	
	
    public String[] generateDDL(SQLObject[] elements, IProgressMonitor progressMonitor) {
    	EngineeringOption[] options = this.getSelectedOptions(elements);
    	String[] statements = this.createSQLObjects(elements, EngineeringOptionID.generateQuotedIdentifiers(options),
    	                      EngineeringOptionID.generateFullyQualifiedNames(options), progressMonitor);
    	if(EngineeringOptionID.generateDropStatement(options)) {
    		String[] drop = this.dropSQLObjects(elements, EngineeringOptionID.generateQuotedIdentifiers(options),
    	                    EngineeringOptionID.generateFullyQualifiedNames(options), progressMonitor);    	
            String[] temp = statements;
            statements = new String[temp.length + drop.length];
            for(int i=0; i<drop.length; ++i) {
                statements[i] = drop[i];
            }
            for(int i=0; i<temp.length; ++i) {
                statements[i+drop.length] = temp[i];
            }
        }
        return statements;
    }

    public String[] createSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
    	return this.createStatements(elements,quoteIdentifiers,qualifyNames,progressMonitor,100);
    }
    	
    public String[] dropSQLObjects(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor) {
    	return this.dropStatements(elements,quoteIdentifiers,qualifyNames,progressMonitor,100);
    }
    
    private String[] createStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        DerbyDdlScript script = new DerbyDdlScript();
        if(this.builder == null) {
            this.builder = new DerbyDdlBuilder();
        }

        Set set = null;
        if (optionDependency != null)
            set = optionDependency.getElementsToUse();

        Iterator it = null;
        if (set != null && set.size() > 0)
            it = optionDependency.getElementsToUse().iterator();
        else
            it = GenericDdlGenerationOptions.getAllContainedDisplayableElementSet(elements).iterator();
        
        EngineeringOption[] options = this.getSelectedOptions(elements);

        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof BaseTable) {
            	if (!EngineeringOptionID.generateTables(options)) continue;
                String statement = builder.createTable((BaseTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!EngineeringOptionID.generateViews(options)) continue;
                String statement = builder.createView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateViewStatement(statement);
            }
            else if(o instanceof Procedure) {
            	if (!EngineeringOptionID.generateStoredProcedures(options)) continue;
                String statement = builder.createProcedure((Procedure) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateRoutineStatement(statement);
            }
            else if(o instanceof UserDefinedFunction) {
            	if (!EngineeringOptionID.generateFunctions(options)) continue;
                String statement = builder.createUserDefinedFunction((UserDefinedFunction) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateRoutineStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!EngineeringOptionID.generateTriggers(options)) continue;
                String statement = builder.createTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
            	if (!EngineeringOptionID.generateCKConstraints(options)) continue;
            	String statement = builder.addCheckConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
            	if(!EngineeringOptionID.generatePKConstraints(options)) continue;
                String statement = builder.addUniqueConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
            	if(!EngineeringOptionID.generateFKConstraints(options)) continue;
                String statement = builder.addForeignKey((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableAddForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!EngineeringOptionID.generateIndexes(options)) continue;
                String statement = builder.createIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addCreateIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    private String[] dropStatements(SQLObject[] elements, boolean quoteIdentifiers, boolean qualifyNames, IProgressMonitor progressMonitor, int task) {
        DerbyDdlScript script = new DerbyDdlScript();
        if(this.builder == null) {
            this.builder = new DerbyDdlBuilder();
        }
        Iterator it = GenericDdlGenerationOptions.getAllContainedDisplayableElementSet(elements).iterator();
        EngineeringOption[] options = this.getSelectedOptions(elements);

        while(it.hasNext()) {
            Object o = it.next();
            if(o instanceof  BaseTable) {
            	if (!EngineeringOptionID.generateTables(options)) continue;
                String statement = builder.dropTable((BaseTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTableStatement(statement);
            }
            else if(o instanceof ViewTable) {
            	if (!EngineeringOptionID.generateViews(options)) continue;
                String statement = builder.dropView((ViewTable) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropViewStatement(statement);
            }
            else if(o instanceof Procedure) {
            	if (!EngineeringOptionID.generateStoredProcedures(options)) continue;
                String statement = builder.dropProcedure((Procedure) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropRoutineStatement(statement);
            }
            else if(o instanceof UserDefinedFunction) {
            	if (!EngineeringOptionID.generateFunctions(options)) continue;
                String statement = builder.dropFunction((UserDefinedFunction) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropRoutineStatement(statement);
            }
            else if(o instanceof Trigger) {
            	if (!EngineeringOptionID.generateTriggers(options)) continue;
                String statement = builder.dropTrigger((Trigger) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropTriggerStatement(statement);
            }
            else if(o instanceof CheckConstraint) {
            	if (!EngineeringOptionID.generateCKConstraints(options)) continue;
                String statement = builder.dropTableConstraint((CheckConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof UniqueConstraint) {
            	if(!EngineeringOptionID.generatePKConstraints(options)) continue;
                String statement = builder.dropTableConstraint((UniqueConstraint) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropConstraintStatement(statement);
            }
            else if(o instanceof ForeignKey) {
            	if(!EngineeringOptionID.generateFKConstraints(options)) continue;
                String statement = builder.dropTableConstraint((ForeignKey) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addAlterTableDropForeignKeyStatement(statement);
            }
            else if(o instanceof Index) {
            	if (!EngineeringOptionID.generateIndexes(options)) continue;
                String statement = builder.dropIndex((Index) o, quoteIdentifiers, qualifyNames);
                if(statement != null) script.addDropIndexStatement(statement);
            }
        }
        return script.getStatements();
    }
    
    public EngineeringOption[] getOptions() {
        if(this.options == null) {
            ResourceBundle resource = ResourceBundle.getBundle("org.eclipse.wst.rdb.internal.core.rte.fe.GenericDdlGeneration"); //$NON-NLS-1$
            Vector optionVec = new Vector();
            EngineeringOption[] temp = GenericDdlGenerationOptions.createDDLGenerationOptions(this.getOptionCategories());
            for (int i = 0; i < temp.length; i++ ) {
            	optionVec.add(temp[i]);
            }
            
            EngineeringOptionCategory additional_element =null;
            for (int i = 0; i < categories.length; i++) {
            	if (categories[i].getId().equals(EngineeringOptionCategoryID.GENERATE_ELEMENTS)){
            		additional_element = categories[i];
            	}
            }

            optionVec.add(new EngineeringOption(EngineeringOptionID.GENERATE_STOREDPROCEDURES,resource.getString("GENERATE_STOREDPROCEDURE"), resource.getString("GENERATE_STOREDPROCEDURE_DES"), true,additional_element)); //$NON-NLS-1$ //$NON-NLS-2$
            optionVec.add(new EngineeringOption(EngineeringOptionID.GENERATE_FUNCTIONS,resource.getString("GENERATE_FUNCTION"), resource.getString("GENERATE_FUNCTION_DES"),true,additional_element)); //$NON-NLS-1$ //$NON-NLS-2$

            this.options = new EngineeringOption[optionVec.size()];
            optionVec.copyInto(this.options);
            
        }
        
        return this.options;
    }
    
	public EngineeringOption[] getOptions(SQLObject[] elements) {
	       return getOptions(elements, false);
	}
	     
	public EngineeringOption[] getOptions(SQLObject[] elements, boolean autoDiscovery) {
        EngineeringOptionCategory[] categories_new = this.getOptionCategories();
        
        EngineeringOptionCategory general_options =null;
        EngineeringOptionCategory additional_element =null;
        for (int i = 0; i < categories_new.length; i++) {
          if (categories_new[i].getId().equals(EngineeringOptionCategoryID.GENERATE_OPTIONS)){
              general_options = categories_new[i];
          } else if (categories_new[i].getId().equals(EngineeringOptionCategoryID.GENERATE_ELEMENTS)){
              additional_element = categories_new[i];
          }
        }           

        //Get the main object, the ones that do no have parents.
        //Look for dependency   
        int idx = 0, size = 0;
        optionDependency = new OptionDependency(elements, autoDiscovery);
        Set sOptions = optionDependency.getOptions();

        this.options = new EngineeringOption[sOptions.size()];
        int i = 0;
        for (Iterator it=sOptions.iterator(); it.hasNext(); i++) {
            this.options[i] = GenericDdlGenerationOptions.getEngineeringOption((String)it.next(), general_options, additional_element);
            if (this.options[i] != null && this.options[i].getCategory().getId().equals(EngineeringOptionCategoryID.GENERATE_ELEMENTS)) {
            	idx = i;
            	size++;
            }
        }
        if (size == 1) {
        	EngineeringOption option = this.options[idx];
        	option.setBoolean(true);
        }       
        return this.options;
    }
    
    private static Set getAllContainedDisplayableElementSetDepedency(SQLObject[] elements) {
        SingletonOptionDependency sod = SingletonOptionDependency.getSingletonObject();
        
        Set s = new TreeSet();
        for(int i=0; i<elements.length; ++i) {
            Class key = null;
            if(elements[i] instanceof Database) {
                key = Database.class;
            } else if(elements[i] instanceof Schema) {
            	key = Schema.class;
           	} else if (elements[i] instanceof BaseTable) {
           		key = Table.class;
           	} else if (elements[i] instanceof Index) {
            	key = Index.class;
           	} else if (elements[i] instanceof Procedure) {
            	key = Procedure.class;
           	} else if (elements[i] instanceof UserDefinedFunction) {
            	key = UserDefinedFunction.class;
           	} else if (elements[i] instanceof ViewTable) {
            	key = ViewTable.class;
           	} else if (elements[i] instanceof Trigger) {
            	key = Trigger.class;
           	} else if (elements[i] instanceof UserDefinedType) {
            	key = UserDefinedType.class;
           	} else if (elements[i] instanceof UniqueConstraint) {
            	key = UniqueConstraint.class;
           	} else if(elements[i] instanceof CheckConstraint) {
            	key = CheckConstraint.class;
           	} else if(elements[i] instanceof ForeignKey) {
            	key = ForeignKey.class;
           	}

            try {
            	int mask = sod.getMask(key).intValue();
            	EngineeringOptionID.populateOptions(s, mask);
            } catch (Exception e) {
            	System.err.println("Missing definition for: " + elements[i].getClass().toString());
            	e.printStackTrace();
            }
        }
        return s;
    }
    
    public EngineeringOptionCategory[] getOptionCategories() {
        if(this.categories == null) {
           this.categories = GenericDdlGenerationOptions.createDDLGenerationOptionCategories();
        }
        return this.categories;
    }

    public EngineeringOption[] getSelectedOptions() {
        if (options == null)
            this.getOptions();
        return options;
    }

    public EngineeringOption[] getSelectedOptions(SQLObject[] elements) {
        if (options == null)
            if (optionDependency == null)
                this.getOptions(elements);
            else
                this.getOptions();
        return options;
    }
    
    private String product;
    private String version;
    private EngineeringOption[] options = null;
    private EngineeringOptionCategory[] categories = null;
	private DerbyDdlBuilder builder = null;
	private OptionDependency optionDependency = null;
	
    public class OptionDependency {
        private SQLObject[] elements = null;
        private Set sOptions = new LinkedHashSet();
        private Set sElementsToUse = null;        
        
        public OptionDependency(SQLObject[] elements, boolean autoDiscovery)
        {
            this.elements = elements;
            sElementsToUse = new LinkedHashSet();
            doDiscovery(autoDiscovery);
        }

        private void doDiscovery(boolean autoDiscovery)
        {
              setOption(EngineeringOptionID.GENERATE_FULLY_QUALIFIED_NAME);
              setOption(EngineeringOptionID.GENERATE_QUOTED_IDENTIFIER);
              setOption(EngineeringOptionID.GENERATE_DROP_STATEMENTS);
              setOption(EngineeringOptionID.GENERATE_CREATE_STATEMENTS);

              Set additionalOptions = new TreeSet();
              if (autoDiscovery) {
                  Iterator it = GenericDdlGenerationOptions.getAllContainedDisplayableElementSet(elements).iterator();
                  while(it.hasNext()) {
                      Object o = it.next();
    
                      sElementsToUse.add(o);
                      if (o instanceof BaseTable) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_TABLES);
                      } else if (o instanceof Index) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_INDICES);
                      } else if (o instanceof Procedure) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_STOREDPROCEDURES);
                      } else if (o instanceof UserDefinedFunction) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_FUNCTIONS);
                      } else if (o instanceof ViewTable) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_VIEWS);
                      } else if (o instanceof Trigger) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_TRIGGERS);
                      } else if (o instanceof UniqueConstraint) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_PK_CONSTRAINTS);
                      } else if(o instanceof CheckConstraint) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_CK_CONSTRAINTS);
                      } else if(o instanceof ForeignKey) {
                        additionalOptions.add(EngineeringOptionID.GENERATE_FK_CONSTRAINTS);
                      } else if(o instanceof Column) {
                      }
                  }
                  for (it = additionalOptions.iterator(); it.hasNext(); )
                    setOption((String)it.next());
              } else {
                  sOptions.addAll(DerbyDdlGenerator.getAllContainedDisplayableElementSetDepedency(elements));
              }
        }
        /**
         * @return Returns the option.
         */
        public Set getOptions() {
            return sOptions;
        }
        /**
         * @param option The option to set.
         */
        public void setOption(String option) {
            this.sOptions.add(option);
        }

        /**
         * @return Returns the sElementsToUse.
         */
        public Set getElementsToUse() {
            return sElementsToUse;
        }
    }
    
    public static class SingletonOptionDependency {
		 
        private Map data = new HashMap();
        private static SingletonOptionDependency ref;
        
        private SingletonOptionDependency()
        {
        }
		 
        public static SingletonOptionDependency getSingletonObject()
        {
            if (ref == null) {
                // it's ok, we can call this constructor
                ref = new SingletonOptionDependency();

                //Database
                int mask = EngineeringOptionID.TABLE | EngineeringOptionID.INDEX | 
                         EngineeringOptionID.VIEW | EngineeringOptionID.PROCEDURE | 
                         EngineeringOptionID.USER_DEFINED_FUNCTION | EngineeringOptionID.TRIGGER | 
                         EngineeringOptionID.UNIQUE_CONSTRAINT | EngineeringOptionID.CHECK_CONSTRAINT | 
                         EngineeringOptionID.FOREIGN_KEY;
                ref.data.put(Database.class,new Integer(mask));

                //Schema
                mask = EngineeringOptionID.TABLE | EngineeringOptionID.INDEX | 
                       EngineeringOptionID.VIEW | EngineeringOptionID.PROCEDURE | 
                       EngineeringOptionID.USER_DEFINED_FUNCTION | EngineeringOptionID.TRIGGER | 
                       EngineeringOptionID.UNIQUE_CONSTRAINT | EngineeringOptionID.CHECK_CONSTRAINT | 
                       EngineeringOptionID.FOREIGN_KEY;
                ref.data.put(Schema.class, new Integer(mask));
               
                //Table
                mask = EngineeringOptionID.TABLE | EngineeringOptionID.INDEX | EngineeringOptionID.TRIGGER | 
                         EngineeringOptionID.UNIQUE_CONSTRAINT | EngineeringOptionID.CHECK_CONSTRAINT | 
                         EngineeringOptionID.FOREIGN_KEY;
                ref.data.put(Table.class, new Integer(mask));
                
                //Index
                mask = EngineeringOptionID.INDEX;
                ref.data.put(Index.class, new Integer(mask));

                //Procedure
                mask = EngineeringOptionID.PROCEDURE;
                ref.data.put(Procedure.class, new Integer(mask));

                //UserDefinedFunction
                mask = EngineeringOptionID.USER_DEFINED_FUNCTION;
                ref.data.put(UserDefinedFunction.class, new Integer(mask));
 
                //ViewTable
                mask = EngineeringOptionID.VIEW | EngineeringOptionID.TRIGGER;
                ref.data.put(ViewTable.class, new Integer(mask));

                //Trigger
                mask = EngineeringOptionID.TRIGGER;
                ref.data.put(Trigger.class, new Integer(mask));

                //Sequence
                mask = EngineeringOptionID.SEQUENCE;
                ref.data.put(Sequence.class, new Integer(mask));

                //UserDefinedType
                mask = EngineeringOptionID.USER_DEFINED_TYPE;
                ref.data.put(UserDefinedType.class, new Integer(mask));

                //UniqueConstraint
                mask = EngineeringOptionID.UNIQUE_CONSTRAINT;
                ref.data.put(UniqueConstraint.class, new Integer(mask));

                //ForeignKey
                mask = EngineeringOptionID.FOREIGN_KEY;
                ref.data.put(ForeignKey.class, new Integer(mask));

                //CheckConstraint
                mask = EngineeringOptionID.CHECK_CONSTRAINT;
                ref.data.put(CheckConstraint.class, new Integer(mask));
          }
          return ref;
        }        
       
        /**
         * @return Returns the data.
         */
        public Integer getMask(Class key) {
            return (Integer)data.get(key);
        }
    }
}

