/*******************************************************************************
 * Copyright (c) 2006 Oracle Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Cameron Bateman/Oracle - initial API and implementation
 *    
 ********************************************************************************/

package org.eclipse.jst.jsf.context.symbol.internal.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jst.jsf.context.symbol.plugin.SymbolsPlugin;

final class TypeUtil 
{
    static IType resolveType(final IType owningType, final String typeSignature)
    {
        // if type signature is already resolved then simply look it up
        if (typeSignature.charAt(0) == Signature.C_RESOLVED)
        {
            IType type = null;
            
            try
            {
                type = owningType.getJavaProject().
                           findType(getFullyQualifiedName(typeSignature));
            }
            catch (JavaModelException jme)
            {
                // do nothing; return type == null;
            }
            
            return type;
        }
        
        final String packageName = Signature.getSignatureQualifier(typeSignature);
        final String typeName = Signature.getSignatureSimpleName(typeSignature);
        final String fullName = "".equals(packageName) ? typeName : packageName + "." + typeName;
        
        IType resolvedType = null;
        
        try
        {
            final String[][] resolved = owningType.resolveType(fullName);
    
            if (resolved != null && resolved.length > 0)
            {
                resolvedType = owningType.getJavaProject().findType(resolved[0][0], resolved[0][1]);
            }
            else
            {
                // if no match, try super types
                resolvedType = resolveInParents(owningType, fullName);
            }
        }
        catch (JavaModelException jme)
        {
            //  do nothing; newType == null
        }
        
        return resolvedType;
    }

    private static String resolveSignatureRelative(final IType owningType, final String typeSignature)
    {
        String  adjustedTypeSignature = typeSignature;
        int arrayDepth = 0;
        
        // handle arrays
        if (Signature.getTypeSignatureKind(adjustedTypeSignature) == Signature.ARRAY_TYPE_SIGNATURE)
        {
            arrayDepth = Signature.getArrayCount(adjustedTypeSignature);
            adjustedTypeSignature = 
                Signature.getElementType(adjustedTypeSignature);
        }
        
        // if already fully resolved, return the input
        if (adjustedTypeSignature.charAt(0) == Signature.C_RESOLVED)
        {
            return typeSignature;
        }
        
        final String fullName = getFullyQualifiedName(adjustedTypeSignature);
        
        IType resolvedType = null;
        
        try
        {
            String[][] resolved = owningType.resolveType(fullName);
    
            if (resolved != null && resolved.length > 0)
            {
                resolvedType = owningType.getJavaProject().findType(resolved[0][0], resolved[0][1]);
            }
            else
            {
                resolvedType = resolveInParents(owningType, fullName);
            }
        }
        catch (JavaModelException jme)
        {
            //  do nothing; newType == null
        }

        if (resolvedType != null)
        {
            String  resolvedTypeSignature = 
                Signature.createTypeSignature
                    (resolvedType.getFullyQualifiedName(), true);
            
            // need to add back array depth if there is one
            for (int i = 0; i < arrayDepth; i++)
            {
                resolvedTypeSignature = "[" + resolvedTypeSignature;
            }
            
            return resolvedTypeSignature;
        }

        if (Signature.getTypeSignatureKind(typeSignature) == 
                Signature.CLASS_TYPE_SIGNATURE)
        {
            // TODO: is there a better way to handle a failure to resolve
            // than just garbage out?
            SymbolsPlugin.getPlugin().log(
                    new Status(org.eclipse.core.runtime.IStatus.WARNING, 
                               SymbolsPlugin.getPlugin().getBundle().getSymbolicName()
                               , 0, 
                               "Failed to resolve type: "+typeSignature,
                               null));
        }
        
        return typeSignature;
    }

    static String resolveTypeSignature(final IType owningType, final String typeSignature)
    {
        final int sigKind = Signature.getTypeSignatureKind(typeSignature);
    
        switch (sigKind)
        {
            case Signature.BASE_TYPE_SIGNATURE:
                return typeSignature;
                
            case Signature.ARRAY_TYPE_SIGNATURE:
            {
                final String elementType = Signature.getElementType(typeSignature);
                
                if (Signature.getTypeSignatureKind(elementType) == Signature.BASE_TYPE_SIGNATURE)
                {
                    return typeSignature;
                }
                else
                {
                    final String resolvedElementType = resolveSignatureRelative(owningType, elementType);
                    String resultType = "";
                    for (int i = 0; i < Signature.getArrayCount(typeSignature);i++)
                    {
                        resultType+=Signature.C_ARRAY;
                    }
                    return resultType+resolvedElementType;
                }
            }
            
            case Signature.CLASS_TYPE_SIGNATURE:
                return resolveSignatureRelative(owningType, typeSignature);
    
            default:
                return typeSignature;
        }
    }
    
    static String getSignature(IType type)
    {
        final String fullyQualifiedName = type.getFullyQualifiedName();
        return Signature.createTypeSignature(fullyQualifiedName, true);
    }

    /**
     * @param typeHierarchy
     * @param type
     * @return all methods of the type and it's super types
     */
    static IMethod[] getAllMethods(final ITypeHierarchy typeHierarchy, final IType type)
    {
        final List   methods = new ArrayList();
        final IType[] superTypes = typeHierarchy.getAllSuperclasses(type);
        final IType[] closure = new IType[superTypes.length+1];
        closure[0] = type;
        System.arraycopy(superTypes, 0, closure, 1, superTypes.length);
        
        for (int i = 0; i < superTypes.length; i++)
        {
            try {
                final IType superType = closure[i];
                methods.addAll(Arrays.asList(superType.getMethods()));
            } catch (JavaModelException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }
            
        return (IMethod[]) methods.toArray(new IMethod[0]);
    }
    
    static String resolveMethodSignature(final IType  owner, 
                                         final String unresolvedSignature)
    {
        // get the list of parameters
        final String[] parameters = 
            Signature.getParameterTypes(unresolvedSignature);
        
        for (int i = 0; i < parameters.length; i++)
        {
            // try to full resolve the type
            parameters[i] = resolveSignatureRelative(owner, parameters[i]);
        }
        
        // resolve return type
        final String resolvedReturn = 
            resolveSignatureRelative(owner, 
                                  Signature.getReturnType(unresolvedSignature));
        
        return Signature.createMethodSignature(parameters, resolvedReturn);
    }
    
    private static String getFullyQualifiedName(final String typeSignature)
    {
        final String packageName = Signature.getSignatureQualifier(typeSignature);
        final String typeName = Signature.getSignatureSimpleName(typeSignature);
        return "".equals(packageName) ? typeName : packageName + "." + typeName;
    }
    
    private static IType resolveInParents(IType  childType, String fullyQualifiedName)
                                throws JavaModelException
    {
        IType resolvedType = null;
        
        // not resolved? try the supertypes
        final ITypeHierarchy typeHierarchy =
            childType.newSupertypeHierarchy(new NullProgressMonitor());
        IType[] superTypes = typeHierarchy.getAllSupertypes(childType);
        String[][]   resolved;
        
        LOOP_UNTIL_FIRST_MATCH:
            for (int i = 0; i < superTypes.length; i++)
        {
            IType type = superTypes[i];
            resolved = type.resolveType(fullyQualifiedName);
            
            if (resolved != null && resolved.length > 0)
            {
                resolvedType = childType.getJavaProject().findType(resolved[0][0], resolved[0][1]);
                break LOOP_UNTIL_FIRST_MATCH;
            }
        }

        return resolvedType;
    }
}
