/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.io.IOException;
import java.io.StringReader;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Displays a result set in the output view.
 */
public class ResultTableViewer 
{
   /** Widget to edit a table cell. */
   protected TextCellEditor stringEditor;
   /** The table viewer. */
   protected TableViewer tableViewer;
   /** Our table. */
   protected Table table;
   /** Our table model. */
   protected ResultTableModel tableModel;
   /** Our rows. */
   protected ArrayList tableRows;
   /** Our layout. */
   protected TableLayout layout;
   /** Our parent. */
   protected Composite parent;
   /** Ellipsis text. */
   protected static String BUTTON_TEXT = "...";
   /** A list of controls for columns with long values. */
   protected ArrayList longColumns;
   /** Button number. */
   protected int butNo = 0;
   /** Maximum string length to display without an ellipsis control. */
   protected static final int MAXDISPLAYLENGTH = 60;
   
   /**
    * Constructs a viewer.
    * @param parent Our parent.
    * @param model Our model.
    */
   protected ResultTableViewer(Composite parent, ResultTableModel model) 
   {
      table =
         new Table(
                  parent,
                  SWT.SINGLE
                  | SWT.FULL_SELECTION
                  | SWT.BORDER
                  | SWT.H_SCROLL
                  | SWT.V_SCROLL);
      
      this.parent = parent;
      tableModel = model;
      tableRows = new ArrayList();
      table.setLinesVisible(true);
      table.setHeaderVisible(true);
      
      longColumns = new ArrayList();
      
      tableViewer = new TableViewer(table);
      tableViewer.setContentProvider(new ResultContentProvider());
      tableViewer.setLabelProvider(new ResultLabelProvider());
      tableViewer.setInput(tableRows);
   }
   
   /** Refreshes the table viewer. */
   protected void refresh() 
   {
      tableViewer.refresh();
   }
   
   /** Clears the result set. */
   protected void clearResult() 
   {
      tableRows.clear();
      removeColumns();
      tableViewer.refresh();
   }
   
   /** Removes columns. */
   protected void removeColumns() 
   {
      Iterator iter;
      TableColumn[] columns = table.getColumns();
      int count = table.getColumnCount();
      for (int i = 0; i < count; ++i) {
         columns[i].dispose();
      }
      // Remove column controls	
      if (longColumns != null) {
         Iterator iterLongColumns = longColumns.iterator();
         while (iterLongColumns.hasNext()) {
            LongColumnControls longColCtls =
               (LongColumnControls) iterLongColumns.next();
            
            iter = longColCtls.getEditorCells().iterator();
            while (iter.hasNext()) {
               ((TableEditor) iter.next()).dispose();
            }
            
            iter = longColCtls.getEditorButtons().iterator();
            while (iter.hasNext()) {
               ((Button) iter.next()).dispose();
            }
         }
         longColumns.clear();
         
      }
   }
   
   /** 
    * Gets the table viewer.
    * @return The TableViewer.
    */
   protected TableViewer getTableViewer() 
   {
      return tableViewer;
   }
   
   /**
    * Sets the column names in the table model.
    */
   protected void setColumnNames() 
   {
      if (tableModel != null) 
      {
         setColumnNames(
                  tableModel.getColumnNames(),
                  tableModel.getColumnCount());
      }
   }
   
   /**
    * Sets given column names.
    * @param colNames The names.
    * @param numColumns The number of columns.
    */
   protected void setColumnNames(String[] colNames, int numColumns) 
   {
      TableColumn column;
      layout = new TableLayout();
      for (int i = 0; i < numColumns; i++) 
      {
         column = new TableColumn(table, SWT.NULL);
         column.setText(colNames[i]);
         layout.addColumnData(new ColumnPixelData(80, true));
         
         column.pack();
      }
      table.setLayout(layout);
   }
   
   /**
    * Sets the ellipsis buttons for a cell with a long value.
    * @param i The rable row index.
    * @param items The items in the row.
    * @param longColumnCtls The manager for the ellipsis buttons.
    */
   protected void setColumnButton(
            int i,              // table row index
            TableItem[] items,  // table of ResultTableRow(s)
            LongColumnControls longColumnCtls) 
   {
      TableItem item = items[i];
      TableEditor tableEditor;
      Button colButton;
      
      tableEditor = new TableEditor(table);
      colButton = new Button(table, SWT.PUSH);
      
      colButton.setToolTipText(
               OutputViewPlugin.getString("OV_LOB_COLUMN_TOOLTIP"));
      colButton.setText(BUTTON_TEXT); // + butNo++);
      colButton.setSize(colButton.computeSize(SWT.DEFAULT, SWT.DEFAULT));
      colButton.setData( new Integer( i ) );  // for use in the ButtonListener
      
      tableEditor.horizontalAlignment = SWT.RIGHT;
      tableEditor.minimumWidth =
         colButton.computeSize(SWT.DEFAULT, SWT.DEFAULT).x;
      
      tableEditor.setEditor(colButton, item, longColumnCtls.getColumnIndex());
      
      longColumnCtls.getEditorCells().add(tableEditor);
      longColumnCtls.getEditorButtons().add(colButton);
      
      colButton.addSelectionListener(longColumnCtls.getListener());
      
   }
   
   /**
    * Sets the result set table model. 
    * @param model The result set table model.
    */
   protected void setTableModel(ResultTableModel model) 
   {
      if (tableModel == model)
      {
         return;		
      }
      tableModel = model;
      if (tableModel != null) 
      {
         clearResult();
         ArrayList rows = tableModel.getTableRows();
         if (rows != null) 
         {
            setColumnNames(
                     tableModel.getColumnNames(),
                     tableModel.getColumnCount());
            tableRows.addAll(rows);
            refresh();
            addColumnButtons();
            table.layout(true);
            
            //				// Workaround for LINUX redrawing problems
            //				redrawTable();
            //				
            //				// Add listeners for each table column
            //				ControlAdapter colListener = new ControlAdapter() {
            //					public void controlMoved(ControlEvent e) {
            //						redrawTable();
            //					}
            //					public void controlResized(ControlEvent e) {
            //						redrawTable();
            //					}
            //				};
            //				
            //				TableColumn[] cols = table.getColumns();
            //				for (int i = 0; i < cols.length; i++) {
            //					cols[i].addControlListener(colListener);
            //				}
         }
      }
   }
   
   /** 
    * Adds ellipsis buttons.
    */
   protected void addColumnButtons() 
   {
      LongColumnControls longColumnCtls;
      
      int[] colTypes = tableModel.getColumnTypes();
      int[] colLengths = tableModel.getColumnLengths();
      String[] colTypeNames = tableModel.getColumnTypeNames();
      for (int col = 0; col < colTypes.length; col++) {
         if ( // Char/Binary column longer than max display length
                  ( (colTypes[col] == Types.VARCHAR
                           || colTypes[col] == Types.CHAR
                           || colTypes[col] == Types.LONGVARCHAR
                           || colTypes[col] == Types.LONGVARBINARY) &&
                           colLengths[col] > MAXDISPLAYLENGTH ) ||
                           // Or LOB column  (LOB length is 0 in DB2 390 meta data
                           (colTypes[col] == Types.CLOB
                                    || colTypes[col] == Types.BLOB
									|| colTypes[col] == Types.JAVA_OBJECT)
                                    || (colTypes[col] == ResultTableViewer.getXMLJdbcEnumType()
                                    		&& "XML".equalsIgnoreCase(colTypeNames[col]))) {
            
            
            longColumnCtls = new LongColumnControls(col);
            longColumns.add(longColumnCtls);
            
            TableItem[] items = table.getItems();
            for (int i = 0; i < items.length; i++) {
               ResultTableRow rtr = (ResultTableRow)items[i].getData();
               int len = rtr.getColumnText(col).length();
               if( len > MAXDISPLAYLENGTH || 
            		   colTypes[col] == ResultTableViewer.getXMLJdbcEnumType()) {
                  setColumnButton(i,items,longColumnCtls);
               }
            }
         }
      }
   }
   
   /**
    * Gets the table model
    * @return the current table model
    */
   public ResultTableModel getTableModel()
   {
	   return tableModel;
   }
   
   /**
    * parses the XML string
    * @param xmlText the xml text to parse
    * @return the DOM document resulted from the parse
    */
   public Document parseXMLText(String xmlText) 
   {
     Document domDocument = null;;
     try
       {
         DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
         DocumentBuilder builder = factory.newDocumentBuilder();
         StringReader reader = new StringReader(xmlText);
         InputSource xmlSource = new InputSource(reader);
         domDocument = builder.parse(xmlSource);                     
       }
       catch (ParserConfigurationException pce)
       {
        // ignore
       }
       catch (SAXException sxe)
       {
         // ignore
       }
       catch (IOException ioe)
       {
         //ignore
       }        
       return domDocument;          
   }
   
   /**
    * Gets the jdbc enum type for XML Type.
    * <p>
    * This is a temporary solution until java.sql.Types defines a type for XML
    * @return the jdbc enum type for XML data type
    */
   public static int getXMLJdbcEnumType() {
      return Types.OTHER;	
   }
   
   //	protected void redrawTable() {
   //	// Fix for LINUX redrawing problem
   //				table.layout();
   //				table.getParent().layout();
   //				table.setSize(table.getSize().x+1, table.getSize().y);
   //				table.setSize(table.getSize().x-1, table.getSize().y);
   //	}
   
   
   /**  Label provider and content provider. */
   protected class ResultContentProvider implements IStructuredContentProvider 
   {
      /**
       * Gets the elements in the table.
       * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
       */
      public Object[] getElements(Object tableRows) {
         ArrayList rows = (ArrayList) tableRows;
         //int num = rows.size();
         Object[] list = rows.toArray();
         return rows.toArray();
      }
      
      /** Disposes. Does nothing. */
      public void dispose() {
      }
      /** Handles inputChanged. Does nothing. */
      public void inputChanged(Viewer viewer, Object old, Object newobj) {
      }
      /** Returns false. */
      public boolean isDeleted(Object object) {
         return false;
      }
   }
   
   /** Label provider. */
   protected class ResultLabelProvider
   extends LabelProvider
   implements ITableLabelProvider 
   {
      public String getColumnText(Object object, int columnIndex) {
         if (object instanceof ResultTableRow) {
            // Return a length of MAXDISPLAYLENGTH for long columns
            // because in Linux the /n actually creates a new line in the table column
            // instead of a funny character.  [wsdbu00048762]
            String columnText = ((ResultTableRow) object).getColumnText(columnIndex);
            if (columnText != null && columnText.length() > MAXDISPLAYLENGTH)
            {
            	return columnText.substring(0, MAXDISPLAYLENGTH);
            }
            return columnText;
         }
         return "";
      }
      
      public Image getColumnImage(Object object, int columnIndex) {
         return null;
      }
   }
   
   /** Ellipsis button listener. */
   protected class ButtonListener extends SelectionAdapter 
   {
      /**
       * Handles selection of the button.
       * @param e The selection event. 
       */
      public void widgetSelected(SelectionEvent e) {
         LongColumnControls longColCtls = null;
         // when the button is clicked, get the text for this column and display it in
         //   a read only dialog.
         
         // Determine which column this listener is for
         Iterator iterLongColumns = longColumns.iterator();
         while (iterLongColumns.hasNext()) {
            longColCtls = (LongColumnControls) iterLongColumns.next();
            if (longColCtls.getListener() == this)
               break;
         }
         Button button = (Button)e.widget;
         //int i = longColCtls.getEditorButtons().indexOf(e.getSource());
         int i = ((Integer)button.getData()).intValue();  // when the button was constructed, the row in which
         // it is placed is cached here.
         ResultTableRow row =
            (ResultTableRow) tableModel.getTableRows().get(i);
         
         String colNames[] = tableModel.getColumnNames();
         int[] colTypes = tableModel.getColumnTypes();
         boolean truncate= false;
         
         // support launching of view for XML data type
         if (colTypes[longColCtls.getColumnIndex()] == ResultTableViewer.getXMLJdbcEnumType()) {
            String title = colNames[longColCtls.getColumnIndex()]+
                " ("+i+","+longColCtls.getColumnIndex()+")";
            
            OutputUtil.launchXMLDialog(parent.getShell(),
            		title, row.getXMLColumnContent(longColCtls.getColumnIndex()));
            
            return;
         }
         else if (colTypes[longColCtls.getColumnIndex()] == Types.CLOB ||
                colTypes[longColCtls.getColumnIndex()] == Types.BLOB ||
                colTypes[longColCtls.getColumnIndex()] == Types.VARCHAR)                
         {
         	// Parses the text to see if it is a xml document            
            Document doc = parseXMLText(row.getColumnText(longColCtls.getColumnIndex()));
            if (doc != null)
            {
            	// Launch XML view since it is a valid XML document
                String title = colNames[longColCtls.getColumnIndex()]+
                    " ("+i+","+longColCtls.getColumnIndex()+")";
                OutputUtil.launchXMLDialog(parent.getShell(),
                		title, row.getColumnText(longColCtls.getColumnIndex()));
            }
            else
            {
            	// launch the default long field dialog
                truncate = true;
                String title = colNames[longColCtls.getColumnIndex()]+" ("+i+","+longColCtls.getColumnIndex()+")";      
                DialogShowLongField ddlDialog =
                   new DialogShowLongField(
                            parent.getShell(),
                            title,
                            row.getColumnText(longColCtls.getColumnIndex()),
                            truncate);                
                
                ddlDialog.open();
            }
            return;
         }
         // end xml support
         
         if (colTypes[longColCtls.getColumnIndex()] == Types.LONGVARBINARY ||
                  colTypes[longColCtls.getColumnIndex()] == Types.BLOB)                              
            truncate = true;
         String title = colNames[longColCtls.getColumnIndex()]+" ("+i+","+longColCtls.getColumnIndex()+")";		
         DialogShowLongField ddlDialog =
            new DialogShowLongField(
                     parent.getShell(),
                     title,
                     row.getColumnText(longColCtls.getColumnIndex()),
                     truncate);         
         
         ddlDialog.open();          
      }      
      
   }
   
   /** Manages ellipsis buttons for columns with long values. */
   protected class LongColumnControls 
   {
      /** Our editor cell list. */
      protected ArrayList editorCells;
      /** Our editor button list. */
      protected ArrayList editorButtons;
      /** Our button listener. */
      protected ButtonListener listener;
      /** Our column index. */
      protected int colIndex;
      /**
       * Constructs a manager for column controls for a given column.
       * @param aColIndex The column index.
       */
      public LongColumnControls(int aColIndex) 
      {
         colIndex = aColIndex;
         editorCells = new ArrayList();
         editorButtons = new ArrayList();
         listener = new ButtonListener();
      }
      /** @return Our editor cell list. */
      public ArrayList getEditorCells() {
         return editorCells;
      }
      /** @return Our editor button list. */
      public ArrayList getEditorButtons() {
         return editorButtons;
      }
      /** @return Our button listener. */
      public ButtonListener getListener() {
         return listener;
      }
      /** @return Our column index. */
      public int getColumnIndex() {
         return colIndex;
      }
   }
}

