/*******************************************************************************
* Copyright (c) 2000, 2004 IBM Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     IBM Corporation - initial API and implementation
*******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.printing.PrintDialog;
import org.eclipse.swt.printing.Printer;
import org.eclipse.swt.printing.PrinterData;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;

/**
 * This class provides a menu and handles the actions that are specific to the
 * Result area of the output view.
 */
public class OVResultAreaMenu implements MenuListener, SelectionListener
{
   /** Our result set table. */
   protected Table myResult;
   /** Our menu. */
   protected Menu myMenu;
   /** StyledText used to support printing. */
   protected StyledText styledText; // used for text print support
   /** Our save menu item. */
   protected MenuItem saveItem;
   /** Our print menu item. */
   protected MenuItem printItem;
   /** Our menu separator. */
   protected MenuItem menuSeparator1;
   /** ResultTableViewer to retrieve the resultTableModel */
   protected ResultTableViewer myResultViewer;
   
   /**
    * Constructs an instance of this class. After creating an instance, a user
    * must call getMenu() to put the menu to a control.
    * @param aTable The result table of output view
    */
   public OVResultAreaMenu(Table aTable)
   {
      myResult = aTable;
      myMenu = new Menu(myResult);
      saveItem = new MenuItem(myMenu, SWT.NONE); // Save Output
      saveItem.setText(OutputViewPlugin.getString("OV_MESSAGE_SAVE_AS"));
      menuSeparator1 = new MenuItem(myMenu, SWT.SEPARATOR);// separator
      printItem = new MenuItem(myMenu, SWT.NONE); // Print
      printItem.setText(OutputViewPlugin.getString("OV_MESSAGE_PRINT"));
      listenAll();
   }
   
   /**
    * Adds listeners for the menu and its items.
    */
   private void listenAll()
   {
      myMenu.addMenuListener(this);
      saveItem.addSelectionListener(this);
      printItem.addSelectionListener(this);
   }
   
   /**
    * Gets the menu.
    * User of this class must call this method to assign this menu to a control.
    * @return The Menu object.
    */
   public Menu getMenu()
   {
      return myMenu;
   }
   
   /**
    * Sets the ResultTableViewer
    * @param viewer the ResultTableViewer
    */
   public void setResultTableViewer(ResultTableViewer viewer)
   {
	   myResultViewer = viewer;
   }
   
   /**
    * Launches file chooser to let a user save results to a file.
    */
   private void saveOutput()
   {
      FileDialog fileDialog = new FileDialog(myResult.getShell(), SWT.SAVE);
      String fileName = fileDialog.open();
      if (fileName != null)
      {
         try
         {
            File file = new File(fileName);
            if (file.isDirectory())
            {
               displayDirectoryError(myResult.getShell(), fileName);
               // return is needed because displayDirectoryError(...) makes
               // a recursive call to this method.
               return;
            }
            if (file.exists())
            {
               // launch question dialog
               Object[] obj = {fileName};
               String message = OutputViewPlugin.getString(
                        "OV_MESSAGE_FILE_EXISTS_DESC", obj);
               MessageBox box = new MessageBox(myResult.getShell(),
                        SWT.YES | SWT.NO);
               box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_EXISTS_TITLE"));
               box.setMessage(message);
               if (box.open() != SWT.YES)
               {
                  return;
               }
            }
            // Use the encoding specified by the user in Window->Preferences
            // to save the file   
            FileOutputStream fos = new FileOutputStream(file);
            OutputStreamWriter outstream = null;
            String encoding = OutputUtil.getCharacterEncoding();
            if (encoding != null && !encoding.equals(""))
            {
            	outstream = new OutputStreamWriter(fos, encoding);
            }
            else
            {
            	outstream = new OutputStreamWriter(fos);
            }
            BufferedWriter bw = new BufferedWriter(outstream);            
            
            int columnCount = myResult.getColumnCount();
            // save table column names
            for (int i=0;i<columnCount;i++)
            {
               bw.write(myResult.getColumn(i).getText());
               bw.write('\t');
            }
            bw.newLine();
            // save table data
            int itemCount = myResult.getItemCount();
            for (int i=0;i<itemCount;i++)
            {           	
            	String rowData = getDataForRow(i);
            	if (rowData != null)
            	{
            		bw.write(rowData); 
            	}
            	else // use data from outputview instead
            	{
            		for (int j=0;j<columnCount;j++)
                    { 
            			bw.write(myResult.getItem(i).getText(j));
            			bw.write('\t');
                    }
            	}            	
            	bw.newLine();
            }
            bw.flush();
            bw.close();
         }
         catch (IOException ex)
         {
            MessageBox box = new MessageBox(myResult.getShell(),
                    SWT.ICON_ERROR);
            box.setText(OutputViewPlugin.getString("OV_STATUS_ERROR"));
            box.setMessage(ex.getMessage());
            box.open();            
         }
      }
   }
   
   /**
    * Retrieve one row of data from ResultTableModel 
    * @param index the index of the row to retrieve the data for
    * @return row data in form of a string, or null if row is <0 or no data can
    * be retrieved.
    * @exception IOException if problem occurs
    */
   private String getDataForRow(int index) throws IOException
   {
	   if (index  > -1 && myResultViewer != null)
	   {
		   ResultTableModel model = myResultViewer.getTableModel();
		   if (model == null)
		   {
			   return null;
		   }
		   ResultTableRow currentRow = model.getRow(index);
		   int columns = model.getColumnCount();		   
		   StringBuffer sb = new StringBuffer(500);
		   for (int i=0;i<columns;i++)
		   {
			   Object obj = currentRow.getValueForColumn(i);
			   if (obj instanceof IXMLResult)
			   {
				   sb.append(((IXMLResult)obj).getEntireDocument());
			   }
			   else
			   {
				   sb.append((String)obj);
			   }
			   sb.append('\t');
		   }
		   return sb.toString();
	   }
	   return null;
   }   
   
   /**
    * Displays an error message box to let a user know that the file that is selected
    * is a directory.  This is needed for Linux because the fileChooser allows
    * the directory name as a file.
    * @param aShell the parent shell
    * @param aFile the file name
    */
   protected void displayDirectoryError(Shell aShell, String aFile)
   {
      // display error message because a directory was chosen
      Object[] obj = {aFile};
      String message = OutputViewPlugin.getString(
               "OV_MESSAGE_FILE_ISDIRECTORY_DESC", obj);
      MessageBox box = new MessageBox(aShell, SWT.ICON_ERROR | SWT.OK);
      box.setText(OutputViewPlugin.getString("OV_MESSAGE_FILE_ISDIRECTORY_TITLE"));
      box.setMessage(message);
      box.open();
      // reopen save dialog again
      saveOutput();
   }
   
   /**
    * Launches print dialog to print Result area of output view.
    * This method uses a StyledText object to take advantage of its text printing
    * support.
    */
   protected void printOutput()
   {
      PrintDialog dialog = new PrintDialog(myResult.getShell(), SWT.NONE);
      PrinterData printerData = dialog.open();
      if (printerData != null)
      {
         Printer printer = new Printer(printerData);
         StringBuffer sb = new StringBuffer(2000);
         int columnCount = myResult.getColumnCount();
         String lineSeparator = System.getProperty("line.separator");
         // column data
         for (int i=0;i<columnCount;i++)
         {
            sb.append(myResult.getColumn(i).getText());
            sb.append('\t');
         }
         sb.append(lineSeparator);
         // table data
         int itemCount = myResult.getItemCount();
         for (int i=0;i<itemCount;i++)
         {
            for (int j=0;j<columnCount;j++)
            {
               sb.append(myResult.getItem(i).getText(j));
               sb.append('\t');
            }
            sb.append(lineSeparator);
         }
         if (styledText == null)
         {
            styledText = new StyledText(myResult, SWT.WRAP);
            styledText.setVisible(false);
         }
         styledText.setText(sb.toString());
         styledText.print(printer).run();
         printer.dispose();
      }
   }
   
   /**
    * Handes the event when the menu is hidden. Does nothing.
    * @param anEvent The MenuEvent.
    */
   public void menuHidden(MenuEvent anEvent)
   {
      // ignore
   }
   
   /**
    * Handes the event when the menu is shown.
    * @param anEvent The MenuEvent
    */
   public void menuShown(MenuEvent anEvent)
   {
      // disable Save Output if no text in area
      saveItem.setEnabled(myResult.getItemCount() > 0);
      
      // disable Print if no text in area
      printItem.setEnabled(myResult.getItemCount() > 0);
   }
   
   /**
    * Handes the event when the default selection occurs. Does nothing.
    * @param anEvent The SelectionEvent
    */
   public void widgetDefaultSelected(SelectionEvent anEvent)
   {
      // ignore
   }
   
   /**
    * Handes the event when the selection of a menuItem occurs.
    * @param anEvent The SelectionEvent
    */
   public void widgetSelected(SelectionEvent anEvent)
   {
      Object source = anEvent.getSource();
      // launch file chooser
      if (source == saveItem)
      {
         saveOutput();
      }
      // launch print dialog
      else if (source == printItem)
      {
         printOutput();
      }
   }
}