/*******************************************************************************
 * Copyright (c) 2008 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0, which accompanies this distribution
 * and is available at http://www.eclipse.org/legal/epl-v10.html.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.ui.internal.menus;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jpt.core.JpaPlatform;
import org.eclipse.jpt.core.JpaStructureNode;
import org.eclipse.jpt.ui.JpaPlatformUi;
import org.eclipse.jpt.ui.JptUiPlugin;
import org.eclipse.jpt.ui.details.MappingUiProvider;
import org.eclipse.jpt.ui.internal.jface.ImageImageDescriptor;
import org.eclipse.jpt.utility.internal.CollectionTools;
import org.eclipse.jpt.utility.internal.iterators.TransformationIterator;
import org.eclipse.ui.ISources;
import org.eclipse.ui.actions.CompoundContributionItem;
import org.eclipse.ui.handlers.IHandlerService;
import org.eclipse.ui.menus.CommandContributionItem;
import org.eclipse.ui.menus.CommandContributionItemParameter;
import org.eclipse.ui.menus.IWorkbenchContribution;
import org.eclipse.ui.services.IServiceLocator;

/**
 * This menu contribution is responsible to populate the Map As menu with the
 * registered mapping types defined in the <code>JptPlatformUi</code>.
 *
 * @see JpaPlatform
 * @see JpaPlatformUi
 *
 * @version 2.0
 * @since 2.0
 */
@SuppressWarnings("nls")
public abstract class MapAsContribution extends CompoundContributionItem
	implements IWorkbenchContribution
{
	/**
	 * Keeps track of the <code>IServiceLocator</code> which is used to retrieve
	 * various services required for invoking the <code>PersistentAttributeMapAsHandler</code>.
	 */
	private IServiceLocator serviceLocator;
	
	
	/**
	 * Creates a new <code>PersistentAttributeMapAsContribution</code>.
	 */
	public MapAsContribution() {
		super();
	}
	
	
	public void initialize(IServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}
		
	@Override
	protected IContributionItem[] getContributionItems() {
		// Retrieve the selection from the handler service
		// which should be an IStructuredSelection of JpaStructureNodes
		IHandlerService handlerService = 
			(IHandlerService) serviceLocator.getService(IHandlerService.class);
		IStructuredSelection currentSelection = 
			(IStructuredSelection) handlerService.getCurrentState().getVariable(ISources.ACTIVE_CURRENT_SELECTION_NAME);
		
		// Assume that all nodes are in the same project (which is very safe)
		// and retrieve the mapping UI providers just from the first item
		JpaStructureNode node = (JpaStructureNode) currentSelection.getFirstElement();
		
		return CollectionTools.array(
			new TransformationIterator<MappingUiProvider<?>, IContributionItem>(mappingUiProviders(node)) {
				@Override
				protected IContributionItem transform(MappingUiProvider<?> next) {
					return contributionItem(next);
				}
			},
			new IContributionItem[0]);
	}
	
	/**
	 * Retrieves the registered {@link MappingUiProvider}s from the given node, 
	 * using its {@link JpaPlatformUi}.
	 *
	 * @param node A test node to determine the {@link JpaPlatformUi} and type 
	 * of providers to return
	 * @return The list of registered {@link MappingUiProvider}s
	 */
	protected Iterator<? extends MappingUiProvider<?>> 
			mappingUiProviders(JpaStructureNode node) {
		JpaPlatform jpaPlatform = node.getJpaProject().getJpaPlatform();
		JpaPlatformUi jpaPlatformUi = JptUiPlugin.getPlugin().jpaPlatformUi(jpaPlatform);
		return mappingUiProviders(jpaPlatformUi, node);
	}
	
	/**
	 * Retrieves the registered {@link MappingUiProvider}s from the given 
	 * {@link JpaPlatformUi} and {@link JpaStructureNode} (to determine type of 
	 * mapping providers to retrieve).
	 *
	 * @param jpaPlatformUi The active {@link JpaPlatformUi} from where the
	 * provider can be retrieved
	 * @param node A test node to determine type of providers to return
	 * @return The list of registered {@link MappingUiProvider}s
	 */
	protected abstract Iterator<? extends MappingUiProvider<?>> 
			mappingUiProviders(JpaPlatformUi platformUi, JpaStructureNode node);
	
	protected IContributionItem contributionItem(MappingUiProvider<?> mappingUiProvider) {
		CommandContributionItem item = 
			new CommandContributionItem(parameter(mappingUiProvider));
		return item;
	}
	
	protected CommandContributionItemParameter parameter(MappingUiProvider<?> mappingUiProvider) {
		CommandContributionItemParameter parameter =
			new CommandContributionItemParameter(
					serviceLocator, 
					createCommandContributionItemId(mappingUiProvider),
					commandId(),
					CommandContributionItem.STYLE_CHECK);
		parameter.label = mappingUiProvider.getLabel();
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(commandParameterId(), mappingUiProvider.getMappingKey());
		parameter.parameters = parameters;
		parameter.icon = new ImageImageDescriptor(mappingUiProvider.getImage());
		parameter.visibleEnabled = true;
		return parameter;
	}
	
	/**
	 * Retrieves the unique identifier of the command that will be invoked for
	 * changing the mapping type of the selected nodes.
	 *
	 * @return The unique identifier of the "map as" command
	 */
	protected abstract String commandId();
	
	/**
	 * Retrieves the unique identifier of the mapping key command parameter that 
	 * will be used for the new mapping type of the selected nodes.
	 *
	 * @return The unique identifier of the "map as" command parameter
	 */
	protected abstract String commandParameterId();
	
	/**
	 * Returns an id for a {@link CommandContributionItem} in the form of 
	 * "<commandId>.<mappingKey>"  
	 * (for example "org.eclipse.jpt.core.ui.persistentTypeMapAs.entity")
	 */
	protected String createCommandContributionItemId(MappingUiProvider<?> mappingUiProvider) {
		return commandId() + "." + mappingUiProvider.getMappingKey();
	}
}
