/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.tests.performance.internal.util;

import java.util.logging.Logger;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.IJobManager;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;

/**
 * @since 3.1
 */
public class EditorTestHelper
{
  /**
   * Runs the event queue on the current display until it is empty.
   */
  public static void runEventQueue()
  {
    IWorkbenchWindow window = getActiveWorkbenchWindow();
    if (window != null)
      runEventQueue(window.getShell());
  }

  public static void runEventQueue(IWorkbenchPart part)
  {
    runEventQueue(part.getSite().getShell());
  }

  public static void runEventQueue(Shell shell)
  {
    runEventQueue(shell.getDisplay());
  }

  public static void runEventQueue(Display display)
  {
    while (display.readAndDispatch())
    {
      // do nothing
    }
  }

  /**
   * Runs the event queue on the current display and lets it sleep until the
   * timeout elapses.
   * 
   * @param millis
   *          the timeout in milliseconds
   */
  public static void runEventQueue(long millis)
  {
    runEventQueue(getActiveDisplay(), millis);
  }

  public static void runEventQueue(IWorkbenchPart part, long millis)
  {
    runEventQueue(part.getSite().getShell(), millis);
  }

  public static void runEventQueue(Shell shell, long millis)
  {
    runEventQueue(shell.getDisplay(), millis);
  }

  public static void runEventQueue(Display display, long minTime)
  {
    if (display != null)
    {
      DisplayHelper.sleep(display, minTime);
    }
    else
    {
      sleep((int)minTime);
    }
  }

  public static IWorkbenchWindow getActiveWorkbenchWindow()
  {
    return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
  }

  public static void forceFocus()
  {
    IWorkbenchWindow window = getActiveWorkbenchWindow();
    if (window == null)
    {
      IWorkbenchWindow[] wbWindows = PlatformUI.getWorkbench().getWorkbenchWindows();
      if (wbWindows.length == 0)
        return;
      window = wbWindows[0];
    }
    Shell shell = window.getShell();
    if (shell != null && !shell.isDisposed())
    {
      shell.forceActive();
      shell.forceFocus();
    }
  }

  public static IWorkbenchPage getActivePage()
  {
    IWorkbenchWindow window = getActiveWorkbenchWindow();
    return window != null ? window.getActivePage() : null;
  }

  public static Display getActiveDisplay()
  {
    IWorkbenchWindow window = getActiveWorkbenchWindow();
    return window != null ? window.getShell().getDisplay() : null;
  }

  public static void joinBackgroundActivities() throws CoreException
  {
    // Join Building
    Logger.global.entering("EditorTestHelper", "joinBackgroundActivities");
    Logger.global.finer("join builder");
    boolean interrupted = true;
    while (interrupted)
    {
      try
      {
        Platform.getJobManager().join(ResourcesPlugin.FAMILY_AUTO_BUILD, null);
        interrupted = false;
      }
      catch (InterruptedException e)
      {
        interrupted = true;
      }
    }
    // Join indexing
    Logger.global.finer("join indexer");
    // make sure we search a concrete name. This is faster according to Kent
    // new SearchEngine().searchAllTypeNames(null, "XXXXXXXXX".toCharArray(), SearchPattern.R_EXACT_MATCH, IJavaSearchConstants.CLASS, SearchEngine.createJavaSearchScope(new IJavaElement[0]), new Requestor(), IJavaSearchConstants.WAIT_UNTIL_READY_TO_SEARCH, null);
    // Join jobs
    joinJobs(0, 0, 500);
    Logger.global.exiting("EditorTestHelper", "joinBackgroundActivities");
  }

  public static boolean joinJobs(long minTime, long maxTime, long intervalTime)
  {
    Logger.global.entering("EditorTestHelper", "joinJobs");
    runEventQueue(minTime);
    DisplayHelper helper = new DisplayHelper()
    {
      public boolean condition()
      {
        return allJobsQuiet();
      }
    };
    boolean quiet = helper.waitForCondition(getActiveDisplay(), maxTime > 0 ? maxTime : Long.MAX_VALUE, intervalTime);
    Logger.global.exiting("EditorTestHelper", "joinJobs", new Boolean(quiet));
    return quiet;
  }

  public static void sleep(int intervalTime)
  {
    try
    {
      Thread.sleep(intervalTime);
    }
    catch (InterruptedException e)
    {
      e.printStackTrace();
    }
  }

  public static boolean allJobsQuiet()
  {
    IJobManager jobManager = Platform.getJobManager();
    Job[] jobs = jobManager.find(null);
    for (int i = 0; i < jobs.length; i++)
    {
      Job job = jobs[i];
      int state = job.getState();
      if (state == Job.RUNNING || state == Job.WAITING)
      {
        Logger.global.finest(job.getName());
        return false;
      }
    }
    return true;
  }

  public static boolean isViewShown(String viewId)
  {
    return getActivePage().findViewReference(viewId) != null;
  }

  public static boolean showView(String viewId, boolean show) throws PartInitException
  {
    IWorkbenchPage activePage = getActivePage();
    IViewReference view = activePage.findViewReference(viewId);
    boolean shown = view != null;
    if (shown != show)
      if (show)
        activePage.showView(viewId);
      else
        activePage.hideView(view);
    return shown;
  }

  public static void bringToTop()
  {
    getActiveWorkbenchWindow().getShell().forceActive();
  }

  public static String showPerspective(String perspective) throws WorkbenchException
  {
    String shownPerspective = getActivePage().getPerspective().getId();
    if (!perspective.equals(shownPerspective))
    {
      IWorkbench workbench = PlatformUI.getWorkbench();
      IWorkbenchWindow activeWindow = workbench.getActiveWorkbenchWindow();
      workbench.showPerspective(perspective, activeWindow);
    }
    return shownPerspective;
  }
}