/**********************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *    Igor Fedorenko & Fabrizio Giustina - Initial API and implementation
 **********************************************************************/
package org.eclipse.jst.server.tomcat.loader;

import java.io.File;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.naming.NamingException;
import javax.naming.directory.Attributes;

import org.apache.naming.NamingEntry;
import org.apache.naming.resources.FileDirContext;

/**
 * Extended FileDirContext implementation that will allow loading of tld files
 * from the META-INF directory (or subdirectories) in classpath. This will fully
 * mimic the behavior of compressed jars also when using unjarred resources.
 * @author Fabrizio Giustina
 */
public class WtpDirContext extends FileDirContext {

    // Map<String, File>
    private Map virtualMappings;

    // Map<String, File>
    private Map tagfileMappings;

    private String virtualClasspath;

    /**
     * Tomcat digester will automatically set this property to the value of the
     * "virtualClasspath" xml attribute.
     * @param path ; separated list of path elements.
     */
    public void setVirtualClasspath(String path) {
        virtualClasspath = path;
    }

    public void allocate() {
        super.allocate();
        virtualMappings = new Hashtable(); // new Hashtable<String, File>();
        tagfileMappings = new Hashtable(); // new Hashtable<String, File>();

        StringTokenizer tkn = new StringTokenizer(virtualClasspath, ";");
        while (tkn.hasMoreTokens()) {
            File file = new File(tkn.nextToken(), "META-INF");

            if (!file.exists() || !file.isDirectory()) {
                continue;
            }
            scanForTlds(file);
        }
    }

    private void scanForTlds(File dir) {

        File[] files = dir.listFiles();
        for (int j = 0; j < files.length; j++) {
            File file = files[j];

            if (file.isDirectory()) {
                scanForTlds(file);
            } else if (file.getName().endsWith(".tld")) {
                virtualMappings.put("~" + System.currentTimeMillis() + "~"
                        + file.getName(), file);
            }
        }

    }

    public void release() {
        super.release();
        virtualMappings = null;
    }

    public Attributes getAttributes(String name) throws NamingException {
        if (name.startsWith("/WEB-INF/") && name.endsWith(".tld")) {
            String tldName = name.substring(name.lastIndexOf("/") + 1);
            if (virtualMappings.containsKey(tldName)) {
                return new FileResourceAttributes((File) virtualMappings
                        .get(tldName));
            }
        } else if (name.startsWith("/META-INF/tags") && name.endsWith(".tag")
                || name.endsWith(".tagx")) {

            // already loaded tag file
            if (tagfileMappings.containsKey(name)) {
                return new FileResourceAttributes((File) tagfileMappings
                        .get(name));
            }

            // unknown tagfile, search for it in virtualClasspath
            StringTokenizer tkn = new StringTokenizer(virtualClasspath, ";");
            while (tkn.hasMoreTokens()) {
                File file = new File(tkn.nextToken(), name);
                if (file.exists()) {
                    tagfileMappings.put(name, file);
                    return new FileResourceAttributes(file);
                }
            }
        }

        return super.getAttributes(name);
    }

    protected Vector list(File file) {
        Vector entries = super.list(file);
        if ("WEB-INF".equals(file.getName())) {
            entries.addAll(getVirtualNamingEntries());
        }
        return entries;
    }

    public Object lookup(String name) throws NamingException {
        if (name.startsWith("/WEB-INF/") && name.endsWith(".tld")) {
            String tldName = name.substring(name.lastIndexOf("/") + 1);
            if (virtualMappings.containsKey(tldName)) {
                return new FileResource((File) virtualMappings.get(tldName));
            }
        } else if (name.startsWith("/META-INF/tags") && name.endsWith(".tag")
                || name.endsWith(".tagx")) {

            // already loaded tag file
            return new FileResource((File) tagfileMappings.get(name));
        }

        return super.lookup(name);
    }

    private List getVirtualNamingEntries() {
        List virtual = new ArrayList();
        for (Iterator iter = virtualMappings.keySet().iterator(); iter
                .hasNext();) {
            String name = (String) iter.next();
            File file = (File) virtualMappings.get(name);
            NamingEntry entry = new NamingEntry(name, new FileResource(file),
                    NamingEntry.ENTRY);
            virtual.add(entry);
        }
        return virtual;
    }

}
