package oracle.adfdemo.view.faces.email;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.mail.Folder;
import javax.mail.MessagingException;
import javax.mail.Part;

import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import javax.faces.context.FacesContext;

import oracle.adf.view.faces.context.AdfFacesContext;

/**
 * Backing bean for the "show message" page. Provides support
 * for downloading attachments, and hides the fact that the "message"
 * comes in processScope from the actual page content.
 */
public class ShowMessageBackingBean
{
  public ShowMessageBackingBean()
  {
    // Code necessary because of:
    //   https://javaserverfaces.dev.java.net/issues/show_bug.cgi?id=22
    AdfFacesContext afContext = AdfFacesContext.getCurrentInstance();
    setMessage((MessageData) afContext.getProcessScope().get("message"));
  }

  public MessageData getMessage()
  {
    return _message;
  }

  public void setAttachmentToDownload(Part attachment)
  {
    _attachmentToDownload = attachment;
  }
  
  
  public Part getAttachmentToDownload()
  {
    return _attachmentToDownload;
  }


  public void setMessage(MessageData message)
  {
    try
    {
      _LOG.log(Level.INFO, "Displaying message {0}", message.getSubject());
    }
    catch (MessagingException me)
    {
      _LOG.log(Level.SEVERE, "Can't get the subject", me);
    }

    _message = message;
  }

  public String downloadAttachment() throws IOException, MessagingException
  {
    if (_attachmentToDownload == null)
    {
      _LOG.severe("No attachment available");
      return null;
    }

    _message.getMessage().getFolder().open(Folder.READ_ONLY);

    InputStream in = _attachmentToDownload.getInputStream();
    
    FacesContext context = FacesContext.getCurrentInstance();
    // Get the ServletResponse;  nothing on ExternalContext is sufficient
    ServletResponse response = (ServletResponse)
      context.getExternalContext().getResponse();
    response.setContentType(_attachmentToDownload.getContentType());

    // If the size of the attachment is known, pass that on.
    int size = _attachmentToDownload.getSize();
    if (size >= 0)
    {
      response.setContentLength(size);
    }

    if (_LOG.isLoggable(Level.INFO))
      _LOG.info("Downloading content+ [size=" + size +",contentType=" +
                _attachmentToDownload.getContentType() + "]");

    if (response instanceof HttpServletResponse)
    {
      String filename = _attachmentToDownload.getFileName();
      if (filename != null)
      {
        ((HttpServletResponse) response).setHeader(
            "Content-disposition",
            "attachment; filename=\"" + filename + "\"");
      }
    }

    // Pass the text along, 128K at a time.
    try
    {
      OutputStream out = response.getOutputStream();
      try
      {
        byte[] buffer = new byte[131072];
        while (true)
        {
          int count = in.read(buffer);
          if (count < 0)
            break;
          
          out.write(buffer, 0, count);
        }
      }
      // Close up the response
      finally
      {
        // And tell JSF that we handled everything
        context.responseComplete();
        out.close();
      }
    }
    // And make sure the folder got closed
    finally
    {
      _message.getMessage().getFolder().close(false);
    }
    
    return null;
  }
  
  private MessageData _message;
  private Part        _attachmentToDownload;

  static private final Logger _LOG =
    Logger.getLogger(ShowMessageBackingBean.class.getName());
}
