/*******************************************************************************
* Copyright (c) 2008, 2012 Oracle. All rights reserved.
* This program and the accompanying materials are made available under the
* terms of the Eclipse Public License v1.0, which accompanies this distribution
* and is available at http://www.eclipse.org/legal/epl-v10.html.
* 
* Contributors:
*     Oracle - initial API and implementation
*******************************************************************************/
package org.eclipse.jpt.jpa.eclipselink.ui.internal.persistence.options;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import org.eclipse.jpt.common.ui.internal.JptCommonUiMessages;
import org.eclipse.jpt.common.ui.internal.util.SWTUtil;
import org.eclipse.jpt.common.ui.internal.widgets.Pane;
import org.eclipse.jpt.common.utility.internal.CollectionTools;
import org.eclipse.jpt.common.utility.internal.StringConverter;
import org.eclipse.jpt.common.utility.internal.iterators.TransformationIterator;
import org.eclipse.jpt.common.utility.internal.model.value.CompositeListValueModel;
import org.eclipse.jpt.common.utility.internal.model.value.PropertyAspectAdapter;
import org.eclipse.jpt.common.utility.internal.model.value.PropertyListValueModelAdapter;
import org.eclipse.jpt.common.utility.internal.model.value.SimpleCollectionValueModel;
import org.eclipse.jpt.common.utility.internal.model.value.SortedListValueModelAdapter;
import org.eclipse.jpt.common.utility.model.value.CollectionValueModel;
import org.eclipse.jpt.common.utility.model.value.ListValueModel;
import org.eclipse.jpt.common.utility.model.value.PropertyValueModel;
import org.eclipse.jpt.common.utility.model.value.ModifiablePropertyValueModel;
import org.eclipse.jpt.jpa.eclipselink.core.context.persistence.Options;
import org.eclipse.jpt.jpa.eclipselink.core.context.persistence.TargetServer;
import org.eclipse.jpt.jpa.eclipselink.ui.internal.EclipseLinkHelpContextIds;
import org.eclipse.jpt.jpa.eclipselink.ui.internal.EclipseLinkUiMessages;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import com.ibm.icu.text.Collator;

/**
 *  TargetServerComposite
 */
public class TargetServerComposite extends Pane<Options>
{
	/**
	 * Creates a new <code>TargetServerComposite</code>.
	 *
	 * @param parentPane The parent container of this one
	 * @param parent The parent container
	 */
	public TargetServerComposite(
								Pane<? extends Options> parentPane,
	                           Composite parent) {

		super(parentPane, parent);
	}

	private PropertyValueModel<String> buildDefaultTargetServerHolder() {
		return new PropertyAspectAdapter<Options, String>(this.getSubjectHolder(), Options.DEFAULT_TARGET_SERVER) {
			@Override
			protected String buildValue_() {
				return TargetServerComposite.this.getDefaultValue(subject);
			}
		};
	}

	private ListValueModel<String> buildDefaultTargetServerListHolder() {
		return new PropertyListValueModelAdapter<String>(
			this.buildDefaultTargetServerHolder()
		);
	}

	private String buildDisplayString(String targetServerName) {
		switch (TargetServer.valueOf(targetServerName)) {
			case jboss :
				return EclipseLinkUiMessages.TargetServerComposite_jboss;
			case netweaver_7_1 :
				return EclipseLinkUiMessages.TargetServerComposite_netweaver_7_1;
			case none :
				return EclipseLinkUiMessages.TargetServerComposite_none;
			case oc4j :
				return EclipseLinkUiMessages.TargetServerComposite_oc4j;
			case sunas9 :
				return EclipseLinkUiMessages.TargetServerComposite_sunas9;
			case weblogic :
				return EclipseLinkUiMessages.TargetServerComposite_weblogic;
			case weblogic_10 :
				return EclipseLinkUiMessages.TargetServerComposite_weblogic_10;
			case weblogic_9 :
				return EclipseLinkUiMessages.TargetServerComposite_weblogic_9;
			case websphere :
				return EclipseLinkUiMessages.TargetServerComposite_websphere;
			case websphere_6_1 :
				return EclipseLinkUiMessages.TargetServerComposite_websphere_6_1;
			case websphere_7 :
				return EclipseLinkUiMessages.TargetServerComposite_websphere_7;
			default :
				throw new IllegalStateException();
		}
	}

	private Comparator<String> buildTargetServerComparator() {
		return new Comparator<String>() {
			public int compare(String targetServer1, String targetServer2) {
				targetServer1 = buildDisplayString(targetServer1);
				targetServer2 = buildDisplayString(targetServer2);
				return Collator.getInstance().compare(targetServer1, targetServer2);
			}
		};
	}

	private StringConverter<String> buildTargetServerConverter() {
		return new StringConverter<String>() {
			public String convertToString(String value) {
				try {
					TargetServer.valueOf(value);
					value = buildDisplayString(value);
				}
				catch (Exception e) {
					// Ignore since the value is not a TargetServer
				}
				return value;
			}
		};
	}

	private ModifiablePropertyValueModel<String> buildTargetServerHolder() {
		return new PropertyAspectAdapter<Options, String>(this.getSubjectHolder(), Options.TARGET_SERVER_PROPERTY) {
			@Override
			protected String buildValue_() {

				String name = subject.getTargetServer();
				if (name == null) {
					name = TargetServerComposite.this.getDefaultValue(subject);
				}
				return name;
			}

			@Override
			protected void setValue_(String value) {

				if (getDefaultValue(subject).equals(value)) {
					value = null;
				}
				subject.setTargetServer(value);
			}
		};
	}

	private ListValueModel<String> buildTargetServerListHolder() {
		ArrayList<ListValueModel<String>> holders = new ArrayList<ListValueModel<String>>(2);
		holders.add(buildDefaultTargetServerListHolder());
		holders.add(buildTargetServersListHolder());
		return new CompositeListValueModel<ListValueModel<String>, String>(holders);
	}

	private Iterator<String> buildTargetServers() {
		return new TransformationIterator<TargetServer, String>(CollectionTools.iterator(TargetServer.values())) {
			@Override
			protected String transform(TargetServer next) {
				return next.name();
			}
		};
	}

	private CollectionValueModel<String> buildTargetServersCollectionHolder() {
		return new SimpleCollectionValueModel<String>(
			CollectionTools.collection(buildTargetServers())
		);
	}

	private ListValueModel<String> buildTargetServersListHolder() {
		return new SortedListValueModelAdapter<String>(
			buildTargetServersCollectionHolder(),
			buildTargetServerComparator()
		);
	}

	private String getDefaultValue(Options subject) {
		String defaultValue = subject.getDefaultTargetServer();

		if (defaultValue != null) {
			return NLS.bind(
				JptCommonUiMessages.DefaultWithOneParam,
				defaultValue
			);
		}
		return JptCommonUiMessages.DefaultEmpty;
	}

	@Override
	protected void initializeLayout(Composite container) {
		Combo combo = addLabeledEditableCombo(
			container,
			EclipseLinkUiMessages.PersistenceXmlOptionsTab_targetServerLabel,
			this.buildTargetServerListHolder(),
			this.buildTargetServerHolder(),
			this.buildTargetServerConverter(),
			EclipseLinkHelpContextIds.PERSISTENCE_OPTIONS_TARGET_SERVER
		);

		SWTUtil.attachDefaultValueHandler(combo);
	}
}
