/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/language-service/ivy/adapters", ["require", "exports", "tslib", "@angular/compiler-cli/src/ngtsc/shims", "@angular/compiler-cli/src/ngtsc/util/src/typescript", "path", "@angular/language-service/ivy/utils"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.LSParseConfigHost = exports.LanguageServiceAdapter = void 0;
    var tslib_1 = require("tslib");
    var shims_1 = require("@angular/compiler-cli/src/ngtsc/shims");
    var typescript_1 = require("@angular/compiler-cli/src/ngtsc/util/src/typescript");
    var p = require("path");
    var utils_1 = require("@angular/language-service/ivy/utils");
    var LanguageServiceAdapter = /** @class */ (function () {
        function LanguageServiceAdapter(project) {
            this.project = project;
            this.entryPoint = null;
            this.constructionDiagnostics = [];
            this.ignoreForEmit = new Set();
            this.factoryTracker = null; // no .ngfactory shims
            this.unifiedModulesHost = null; // only used in Bazel
            this.templateVersion = new Map();
            this.rootDirs = typescript_1.getRootDirs(this, project.getCompilationSettings());
        }
        LanguageServiceAdapter.prototype.isShim = function (sf) {
            return shims_1.isShim(sf);
        };
        LanguageServiceAdapter.prototype.fileExists = function (fileName) {
            return this.project.fileExists(fileName);
        };
        LanguageServiceAdapter.prototype.readFile = function (fileName) {
            return this.project.readFile(fileName);
        };
        LanguageServiceAdapter.prototype.getCurrentDirectory = function () {
            return this.project.getCurrentDirectory();
        };
        LanguageServiceAdapter.prototype.getCanonicalFileName = function (fileName) {
            return this.project.projectService.toCanonicalFileName(fileName);
        };
        /**
         * Return the real path of a symlink. This method is required in order to
         * resolve symlinks in node_modules.
         */
        LanguageServiceAdapter.prototype.realpath = function (path) {
            var _a, _b, _c;
            return (_c = (_b = (_a = this.project).realpath) === null || _b === void 0 ? void 0 : _b.call(_a, path)) !== null && _c !== void 0 ? _c : path;
        };
        /**
         * readResource() is an Angular-specific method for reading files that are not
         * managed by the TS compiler host, namely templates and stylesheets.
         * It is a method on ExtendedTsCompilerHost, see
         * packages/compiler-cli/src/ngtsc/core/api/src/interfaces.ts
         */
        LanguageServiceAdapter.prototype.readResource = function (fileName) {
            if (utils_1.isTypeScriptFile(fileName)) {
                throw new Error("readResource() should not be called on TS file: " + fileName);
            }
            // Calling getScriptSnapshot() will actually create a ScriptInfo if it does
            // not exist! The same applies for getScriptVersion().
            // getScriptInfo() will not create one if it does not exist.
            // In this case, we *want* a script info to be created so that we could
            // keep track of its version.
            var snapshot = this.project.getScriptSnapshot(fileName);
            if (!snapshot) {
                // This would fail if the file does not exist, or readFile() fails for
                // whatever reasons.
                throw new Error("Failed to get script snapshot while trying to read " + fileName);
            }
            var version = this.project.getScriptVersion(fileName);
            this.templateVersion.set(fileName, version);
            return snapshot.getText(0, snapshot.getLength());
        };
        LanguageServiceAdapter.prototype.isTemplateDirty = function (fileName) {
            var lastVersion = this.templateVersion.get(fileName);
            var latestVersion = this.project.getScriptVersion(fileName);
            return lastVersion !== latestVersion;
        };
        return LanguageServiceAdapter;
    }());
    exports.LanguageServiceAdapter = LanguageServiceAdapter;
    /**
     * Used to read configuration files.
     *
     * A language service parse configuration host is independent of the adapter
     * because signatures of calls like `FileSystem#readFile` are a bit stricter
     * than those on the adapter.
     */
    var LSParseConfigHost = /** @class */ (function () {
        function LSParseConfigHost(serverHost) {
            this.serverHost = serverHost;
        }
        LSParseConfigHost.prototype.exists = function (path) {
            return this.serverHost.fileExists(path) || this.serverHost.directoryExists(path);
        };
        LSParseConfigHost.prototype.readFile = function (path) {
            var content = this.serverHost.readFile(path);
            if (content === undefined) {
                throw new Error("LanguageServiceFS#readFile called on unavailable file " + path);
            }
            return content;
        };
        LSParseConfigHost.prototype.lstat = function (path) {
            var _this = this;
            return {
                isFile: function () {
                    return _this.serverHost.fileExists(path);
                },
                isDirectory: function () {
                    return _this.serverHost.directoryExists(path);
                },
                isSymbolicLink: function () {
                    throw new Error("LanguageServiceFS#lstat#isSymbolicLink not implemented");
                },
            };
        };
        LSParseConfigHost.prototype.pwd = function () {
            return this.serverHost.getCurrentDirectory();
        };
        LSParseConfigHost.prototype.extname = function (path) {
            return p.extname(path);
        };
        LSParseConfigHost.prototype.resolve = function () {
            var paths = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                paths[_i] = arguments[_i];
            }
            return p.resolve.apply(p, tslib_1.__spread(paths));
        };
        LSParseConfigHost.prototype.dirname = function (file) {
            return p.dirname(file);
        };
        LSParseConfigHost.prototype.join = function (basePath) {
            var paths = [];
            for (var _i = 1; _i < arguments.length; _i++) {
                paths[_i - 1] = arguments[_i];
            }
            return p.join.apply(p, tslib_1.__spread([basePath], paths));
        };
        return LSParseConfigHost;
    }());
    exports.LSParseConfigHost = LSParseConfigHost;
});
//# sourceMappingURL=data:application/json;base64,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