import { DebugProtocol } from 'vscode-debugprotocol';
import { ISetBreakpointsArgs, ISetBreakpointsResponseBody, ISetBreakpointResult } from '../debugAdapterInterfaces';
import { Protocol as Crdp } from 'devtools-protocol';
import { ReasonType } from './stoppedEvent';
import { InternalSourceBreakpoint } from './internalSourceBreakpoint';
import { ScriptContainer } from './scripts';
import { ChromeDebugAdapter } from '..';
import { IPendingBreakpoint } from './chromeDebugAdapter';
import { ChromeConnection } from './chromeConnection';
export interface IHitConditionBreakpoint {
    numHits: number;
    shouldPause: (numHits: number) => boolean;
}
/**
 * Encapsulates all the logic surrounding breakpoints (e.g. set, unset, hit count breakpoints, etc.)
 */
export declare class Breakpoints {
    private readonly adapter;
    private readonly _chromeConnection;
    private static SET_BREAKPOINTS_TIMEOUT;
    private static HITCONDITION_MATCHER;
    private _breakpointIdHandles;
    private _nextUnboundBreakpointId;
    private _pendingBreakpointsByUrl;
    private _hitConditionBreakpointsById;
    private _committedBreakpointsByUrl;
    private _setBreakpointsRequestQ;
    readonly committedBreakpointsByUrl: Map<string, ISetBreakpointResult[]>;
    private getValueFromCommittedBreakpointsByUrl(url);
    private setValueForCommittedBreakpointsByUrl(url, value);
    private readonly chrome;
    constructor(adapter: ChromeDebugAdapter, _chromeConnection: ChromeConnection);
    reset(): void;
    /**
     * Using the request object from the DAP, set all breakpoints on the target
     * @param args The setBreakpointRequest arguments from the DAP client
     * @param scripts The script container associated with this instance of the adapter
     * @param requestSeq The request sequence number from the DAP
     * @param ids IDs passed in for previously unverified breakpoints
     */
    setBreakpoints(args: ISetBreakpointsArgs, scripts: ScriptContainer, requestSeq: number, ids?: number[]): Promise<ISetBreakpointsResponseBody>;
    validateBreakpointsPath(args: ISetBreakpointsArgs): Promise<void>;
    /**
     * Makes the actual call to either Debugger.setBreakpoint or Debugger.setBreakpointByUrl, and returns the response.
     * Responses from setBreakpointByUrl are transformed to look like the response from setBreakpoint, so they can be
     * handled the same.
     */
    addBreakpoints(url: string, breakpoints: InternalSourceBreakpoint[], scripts: ScriptContainer): Promise<ISetBreakpointResult[]>;
    /**
     * Adds a single breakpoint in the target using the url for the script
     * @param scriptId the chrome-devtools script id for the script on which we want to add a breakpoint
     * @param urlRegex The regular expression string which will be used to find the correct url on which to set the breakpoint
     * @param lineNumber Line number of the breakpoint
     * @param columnNumber Column number of the breakpoint
     * @param condition The (optional) breakpoint condition
     */
    addOneBreakpointByUrl(scriptId: Crdp.Runtime.ScriptId | undefined, urlRegex: string, lineNumber: number, columnNumber: number, condition: string): Promise<ISetBreakpointResult>;
    /**
     * Using the request object from the DAP, set all breakpoints on the target
     * @param args The setBreakpointRequest arguments from the DAP client
     * @param scripts The script container associated with this instance of the adapter
     * @param requestSeq The request sequence number from the DAP
     * @param ids IDs passed in for previously unverified breakpoints
     */
    getBreakpointsLocations(args: DebugProtocol.BreakpointLocationsArguments, scripts: ScriptContainer, requestSeq: number): Promise<DebugProtocol.BreakpointLocationsResponse['body']>;
    /**
     * Transform breakpoint responses from the chrome-devtools target to the DAP response
     * @param url The URL of the script for which we are translating breakpoint responses
     * @param responses The setBreakpoint responses from the chrome-devtools target
     * @param requestBps The list of requested breakpoints pending a response
     * @param ids IDs passed in for previously unverified BPs
     */
    private targetBreakpointResponsesToBreakpointSetResults(url, responses, requestBps, ids?);
    private addHitConditionBreakpoint(requestBp, response);
    private clearAllBreakpoints(url);
    onBreakpointResolved(params: Crdp.Debugger.BreakpointResolvedEvent, scripts: ScriptContainer): void;
    private generateNextUnboundBreakpointId();
    private unverifiedBpResponse(args, requestSeq, targetScriptUrl, message?);
    private unverifiedBpResponseForBreakpoints(args, requestSeq, targetScriptUrl, breakpoints, defaultMessage?);
    handleScriptParsed(script: Crdp.Debugger.ScriptParsedEvent, scripts: ScriptContainer, mappedUrl: string, sources: string[]): Promise<void>;
    resolvePendingBPs(source: string, scripts: ScriptContainer): Promise<void>;
    resolvePendingBreakpoint(pendingBP: IPendingBreakpoint, scripts: ScriptContainer): Promise<void>;
    handleHitCountBreakpoints(expectingStopReason: ReasonType, hitBreakpoints: any): {
        didPause: boolean;
    };
}
